package ru.yandex.crypta.graph2.model.matching.score;

import java.util.stream.Collectors;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.crypta.graph.engine.proto.TScore;
import ru.yandex.crypta.graph.engine.proto.TStats;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.misc.lang.Check;

@YTreeObject
public class MetricsTree {

    @YTreeField
    private double score;

    @YTreeField
    private MapF<String, Double> children = Cf.map();

    /**
     * Single unnamed metric
     */
    public MetricsTree(double score) {
        this.score = score;
    }

    /**
     * Composite metric
     */
    public MetricsTree(double score, MapF<String, Double> children) {
        this.score = score;
        this.children = children;
    }

    /**
     * Single named metric
     */
    public MetricsTree(double score, String name) {
        this.score = score;
        this.children = Cf.map(name, score);
    }

    public MapF<String, Double> getChildren() {
        return children;
    }

    public void setChildren(MapF<String, Double> children) {
        this.children = children;
    }

    public double getScore() {
        return score;
    }

    public MetricsTree minus(MetricsTree other) {
        Check.notNull(other);

        SetF<String> allMetrics = children.keySet().plus(other.children.keySet());
        MapF<String, Double> metricsDiff = allMetrics.toMap(key -> key, metric -> {
            double thisScore = this.children.getOrElse(metric, 0.0);
            double otherScore = other.children.getOrElse(metric, 0.0);
            return thisScore - otherScore;
        });

        MetricsTree metricsTree = new MetricsTree(this.score - other.score);
        metricsTree.setChildren(metricsDiff);
        return metricsTree;
    }

    public static MetricsTree calculateScoreGain(MetricsTree mergedComponentMetrics,
            CollectionF<MetricsTree> splitMetrics)
    {
        Check.notNull(mergedComponentMetrics, "Must be already calculated");
        splitMetrics.forEach(metricsTree -> {
            Check.notNull(metricsTree, "Must be already calculated");
        });

        MetricsTree oldMaxScore = splitMetrics.maxBy(MetricsTree::getScore);

        return mergedComponentMetrics.minus(oldMaxScore);
    }

    public static MetricsTree buildFromTStats(TStats stats) {
        MapF<String, Double> children = Cf.hashMap();
        children.putAll(
                stats.getScoresList().stream().map(
                        score -> Tuple2.tuple(score.getName(), score.getValue())
                ).collect(Collectors.toList())
        );
        return new MetricsTree(stats.getGeneralScore(), children);
    }

    public TStats convertToTStats() {
        TStats.Builder statsBuilder = TStats.newBuilder();
        statsBuilder.setGeneralScore(score);
        statsBuilder.addAllScores(children.mapValuesWithKey(
                (score, value) -> TScore
                        .newBuilder()
                        .setName(score)
                        .setValue(value)
                        .build()
            ).values()
        );
        return statsBuilder.build();
    }
}

