import os
import time
import statface_client

from jinja2 import Environment, FileSystemLoader
from crypta.lib.python.yql_runner.base_parser import BaseParser


def wait(sleep=5, timeout=60):
    """
    Wait till predicat false, or timeout exceed.
    :param predicat: function to check on each iteration
    :param sleep: time beetween iterations
    :param timeout: max time to wait predicat false
    """
    def inner(predicat):
        def wrapped(*args, **kwargs):
            start = time.time()
            while time.time() - start < timeout:
                time.sleep(sleep)
                if predicat(*args, **kwargs):
                    return
            raise RuntimeError('Timeout exceed')
        return wrapped
    return inner


class Singleton(type):

    _instances = {}

    def __call__(cls, *args, **kwargs):
        if cls not in cls._instances:
            cls._instances[cls] = super(Singleton, cls).__call__(*args, **kwargs)
        return cls._instances[cls]


class StatfaceClient(statface_client.StatfaceClient):

    __metaclass__ = Singleton

    def __init__(self):
        super(StatfaceClient, self).__init__(
            client_config=dict(host=self.get_host(), oauth_token=self.get_token()))

    def get_host(self):
        if os.environ.get('ENV_TYPE', '').lower() == 'production':
            return statface_client.STATFACE_PRODUCTION
        return statface_client.STATFACE_BETA

    def get_token(self):
        return os.environ.get('STAT_TOKEN')


class YqlQuery(BaseParser):

    """ Run YQL query to calc metrics """

    def __init__(self, query_path, date, yt_path, version):
        super(YqlQuery, self).__init__(
            date=date, is_embedded=False,
            yt_proxy='hahn.yt.yandex.net', pool='crypta_graph',
            # binaries for mrjob and udfs set to None (because use yql-client mode)
            mrjob_binary=None, udf_resolver_binary=None, udfs_dir=None)
        self.query_path = query_path
        self.yt_path = yt_path
        self.version = version

    def get_libs(self, **kwargs):
        return []

    def get_dirs(self):
        return {'source': self.yt_path.format(date=self.date)}

    def render_query(self):
        """ Use filesystem environment to allow include """
        base_path, template_name = os.path.split(self.query_path)
        template = Environment(
            loader=FileSystemLoader((base_path, os.path.dirname(base_path)))
        ).get_template(template_name)
        return template.render(**self.get_context_data())

    def get_context_data(self, **kwargs):
        context = super(YqlQuery, self).get_context_data(**kwargs)
        context.update(version=self.version)
        return context
