import argparse
import logging
import os
import time
import yaml

from dateutil.parser import parse as parse_dt
from datetime import datetime, timedelta

import yt.wrapper as yt

from library.python import resource
from crypta.lib.python.jinja_resource import JinjaResourceMixin
from crypta.lib.python.solomon.reporter import create_throttled_solomon_reporter
from crypta.lib.python.yql.client import create_yql_client
from statface_client import StatfaceClient, STATFACE_PRODUCTION, STATFACE_BETA


logger = logging.getLogger("YqlDailyReport")


class StdoutMetricsReporter(object):
    def report(self, metrics):
        for m in metrics:
            print str(m)


class StatfaceMetricsReporter(object):
    def __init__(self, statface_client, report_name, report_scale, report_config):
        self.statface_client = statface_client
        self.report_name = report_name
        self.report_scale = report_scale
        self.report_config = report_config

    def report(self, metrics):
        report = self.statface_client.get_report(self.report_name)
        report_config = self.report_config
        try:
            report_config = report_config.encode("utf-8")
        except UnicodeDecodeError:
            pass
        report.upload_config(report_config)
        report.upload_data(scale=self.report_scale, data=metrics)


class SolomonSensorConfig(object):
    def __init__(self, sensor_key=None, sensor_value=None, sensor_keys=[], labels=[]):
        self.sensor_key = sensor_key
        self.sensor_value = sensor_value
        self.sensor_keys = sensor_keys
        self.labels = labels


class SolomonMetricsReporter(object):
    def __init__(self, client, sensors):
        self.client = client
        self.sensors = sensors

    def report(self, metrics):
        counter = 0
        for row in metrics:
            ts_datetime = parse_dt(row["fielddate"])
            labels = {label_key: row[label_key] for label_key in self.sensors.labels}

            if self.sensors.sensor_key and self.sensors.sensor_value:
                self.client.set_value(
                    sensor=row[self.sensors.sensor_key],
                    value=row[self.sensors.sensor_value],
                    labels=labels,
                    ts_datetime=ts_datetime
                )
                counter += 1
            if self.sensors.sensor_keys:
                for sensor_key in self.sensors.sensor_keys:
                    self.client.set_value(
                        sensor=sensor_key,
                        value=row[sensor_key],
                        labels=labels,
                        ts_datetime=ts_datetime
                    )
                    counter += 1

            if counter % 1000 == 0:
                time.sleep(1)
        time.sleep(3)


class YqlDailyReportBase(object):

    query = ""

    report_config = ""
    report_name = ""
    report_scale = "daily"

    solomon_config = None
    solomon_service = None
    solomon_sensors = None

    def __init__(self, yql_client, report_date):
        self.yql_client = yql_client
        if self.yql_client.pool is None and os.getenv("YT_POOL"):
            self.yql_client.pool = os.getenv("YT_POOL")

        self.report_date = report_date

        self.reporters = []
        self.reporters.append(StdoutMetricsReporter())

        if self.report_config:
            statface_host = env("STATFACE_HOST", STATFACE_PRODUCTION, STATFACE_BETA)
            logger.info("Statface host: {}".format(statface_host))
            statface_client = StatfaceClient(
                host=statface_host,
                client_config={"oauth_token": os.getenv("STATFACE_OAUTH")}
            )
            self.reporters.append(
                StatfaceMetricsReporter(statface_client, self.report_name, self.report_scale, self.report_config)
            )

        if self.solomon_service and self.solomon_sensors:
            cfg = dict()
            cfg.update(self.solomon_config)
            cfg.update(self.solomon_service)

            solomon_client = create_throttled_solomon_reporter(push_interval=1, **cfg)
            self.reporters.append(
                SolomonMetricsReporter(solomon_client, self.solomon_sensors)
            )

    def run_metrics(self):
        assert self.query, "Query must not be empty"
        assert self.report_config or self.solomon_sensors, "Report config must not be empty"
        assert self.report_name or self.solomon_service, "Report name must not be empty"

        yql_res = self.yql_client.execute(self.query, syntax_version=1)

        metrics = []
        if len(yql_res) > 0:
            for _, row in yql_res[0].full_dataframe.iterrows():
                mtr = {}
                mtr.update(row)
                metrics.append(mtr)

        for reporter in self.reporters:
            reporter.report(metrics)

    def get_generate_date(self, yt_path=None):
        return self._get_yt_attribute(yt_path or self.yt_path, "generate_date")

    def get_create_date(self, yt_path=None):
        return self._get_yt_attribute(yt_path or self.yt_path, "creation_time").split("T")[0]

    def _get_yt_attribute(self, yt_path, attribute):
        return yt.get("{0}/@".format(yt_path)).get(attribute)


class ResourceDailyReport(YqlDailyReportBase, JinjaResourceMixin):

    """ Get report and query config from resource """

    QUERY_RESOURCE = "/query.sql.j2"
    REPORT_RESOURCE = "/report.yaml"
    CONFIG_RESOURCE = "/config.yaml"

    @property
    def query(self):
        return self.render(self.QUERY_RESOURCE)

    @property
    def report_config(self):
        return resource.find(self.REPORT_RESOURCE)

    @property
    def yaml_config(self):
        return yaml.load(resource.find(self.CONFIG_RESOURCE))

    @property
    def solomon_config(self):
        return self.yaml_config.get("solomon_config")

    @property
    def solomon_service(self):
        return self.yaml_config.get("solomon_service")

    @property
    def solomon_sensors(self):
        sensors_yaml = self.yaml_config.get("solomon_sensors")
        if sensors_yaml:
            return SolomonSensorConfig(
                sensor_key=sensors_yaml.get('sensor_key'),
                sensor_value=sensors_yaml.get('sensor_value'),
                sensor_keys=sensors_yaml.get('sensor_keys'),
                labels=sensors_yaml.get('labels')
            )
        else:
            return None

    @property
    def report_name(self):
        return self.yaml_config.get("report_name")

    @property
    def yt_path(self):
        return self.yaml_config.get("yt_path")

    def get_context_data(self, **kwargs):
        context = super(ResourceDailyReport, self).get_context_data(**kwargs)
        context["date"] = self.get_date()
        return context

    def get_date(self):
        return {
            "@generate_date": self.get_generate_date,
            "@create_date": self.get_create_date,
            "@daily_generate_date": lambda: self._get_yt_attribute(self.yt_path, "daily_generate_date"),
        }.get(self.report_date, lambda: self.report_date)()


def env(key, prod, testing=None):
    result = os.getenv(key)
    if not result:
        env_type = os.getenv("ENVIRONMENT")
        if env_type in ["stable", "production"]:
            result = prod
        else:
            result = testing or prod

    return result


def daily_report_main(report_class, parse_additional_command_line_params=[]):
    yesterday = (datetime.now() - timedelta(days=1)).strftime("%Y-%m-%d")

    argparser = argparse.ArgumentParser()
    argparser.add_argument("--date", type=str, default=yesterday)

    for param in parse_additional_command_line_params:
        argparser.add_argument("--" + param, type=str)

    args = argparser.parse_args()

    logger.info("Running report {}".format(report_class.__name__))
    logger.info("Date: {}".format(args.date))

    yt_proxy = env("YT_PROXY", "hahn.yt.yandex.net")
    yt.config.set_proxy(yt_proxy)
    logger.info("YT proxy: {}".format(yt_proxy))

    yql = create_yql_client(yt_proxy=yt_proxy, token=os.getenv("YQL_TOKEN"), script_name=report_class.__name__)

    params_dict = vars(args)
    additional_params = {param: params_dict[param] for param in parse_additional_command_line_params}

    report = report_class(yql_client=yql, report_date=args.date, **additional_params)
    report.run_metrics()


def main_wrapper(report_main, report_class):
    def wrapped(*args, **kwargs):
        return report_main(report_class)

    return wrapped
