import argparse
import datetime
import logging
import os
import re
import sys
import yaml

import yt.wrapper as yt
from library.python import resource

from crypta.graph.metrics.stats_base.lib import MetricsProcessor, PredictTimeSeries, upload_to_solomon

V1_VERSION = "v1"
V2_VERSION = "v2"
V2_EXPERIMENT_VERSION = "v2exp"

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
handler = logging.StreamHandler(sys.stdout)
handler.setLevel(logging.DEBUG)
formatter = logging.Formatter('%(asctime)s [%(levelname)s] %(message)s')
handler.setFormatter(formatter)
logger.addHandler(handler)


def make_arg_parser():
    parser = argparse.ArgumentParser(description="Calc base matching metrics on YQL")
    parser.add_argument(
        "-s",
        "--source",
        help="YT path to matching source table "
        "Example: //home/crypta/production/state/graph/v2/matching/vertices_no_multi_profile"
        "Allow {date} placeholder",
        required=True,
    )
    parser.add_argument(
        "--history-dir",
        help="Needed for id's stable metrics. YT path to directory with backups of matching source table"
        "Example: //home/crypta/archive/graph/vertices_no_multi_profile",
        required=False,
        default=""
    )
    parser.add_argument(
        "--export-dir",
        help="YT path to directory with export crypta matching for vulture"
        "Example: //home/crypta/production/state/graph/v2/export/vulture",
        required=False,
        default=""
    )
    parser.add_argument(
        "--debug-dir",
        help="YT path to directory where will be stored artifacts of calculating retargeting metrics"
        "Example: //home/crypta/production/graph/metrics",
        required=False,
        default=""
    )
    parser.add_argument(
        "-d",
        "--date",
        help='Date for metrics calc. Allow "@generate_date" - ' "in this case date will be get from table attributes.",
        required=True,
    )
    parser.add_argument(
        "-v", "--version", help="Matching version", required=True
    )
    parser.add_argument(
        "--local", help="Do not upload report on YT and Statface", action="store_true", dest="local", default=False
    )
    parser.add_argument(
        "--metrics",
        help="Space separated list of metrics to calc. "
        "Filter by prefix in config/metrics.yaml (default: all metrics)",
        nargs="+",
        required=False,
    )
    parser.add_argument(
        "--skip",
        help="Space separated list of metrics to skip. " "Filter by prefix in config/metrics.yaml (default: ads)",
        nargs="+",
        required=False,
        default=["ads", "taxi"],
    )
    return parser


def get_source(args):
    raw_source = args.source
    if raw_source.endswith('{last_ts}'):
        return max(yt.list(raw_source.split('{last_ts}')[0].rstrip('/'), absolute=True), key=lambda tbl: str(tbl))
    return raw_source


def get_date(args, source):
    date = args.date
    if args.date.startswith("@"):
        date = str(yt.get("{path}/{attr}".format(path=source, attr=args.date)))
        assert date, "At table [{path}] should be not empty attribute {attr}".format(path=source, attr=args.date)
    elif args.date == "yesterday":
        date = (datetime.date.today() - datetime.timedelta(days=1)).strftime("%Y-%m-%d")
    assert re.match(r"^\d{4}-\d{2}-\d{2}$", date), "Date should be in format YYYY-MM-DD, got {0!r}".format(date)
    return date


def filter_metrics(args, metrics):
    """ Filter only metrics for calc """
    for metrica in metrics:
        if args.metrics:
            if metrica["prefix"] not in args.metrics:
                logger.info("Skip [whitelist] run metrics for {0[prefix]!r}".format(metrica))
                continue
        elif args.skip:
            if metrica["prefix"] in args.skip:
                logger.info("Skip [blacklist] run metrics for {0[prefix]!r}".format(metrica))
                continue
        logger.info("Allow run metrics for {0[prefix]!r}".format(metrica))
        yield metrica


def get_exp_stats_metrics(exp_stats_metrics):
    testids = exp_stats_metrics["testids"]
    exp_types = {}
    for conf_type in exp_stats_metrics["types"]:
        exp_type = dict(**conf_type)
        for id in ["exp_id", "etalon_exp_id"]:
            exp_type[id] = testids[str(conf_type[id])]
        exp_types[conf_type["name"]] = exp_type

    for metrica in exp_stats_metrics["metrics"]:
        for type_name in metrica.pop("types"):
            yield dict(metrica, **exp_types.get(type_name))


def check_stable_metrics(metrics, history_dir):
    if any(metric["prefix"] == "stable" for metric in metrics) and not history_dir:
        raise Exception("For stable metrics you should pass --history-dir")


def predict(args, prefixes, metrics_conf, storage_conf):
    last_exception = None

    for predicts in metrics_conf["prophets"]:
        predicts["version"] = predicts["version"] or args.version
        if (predicts["kind"] not in prefixes) or (predicts["version"] != args.version):
            # do nothing
            continue
        log_predict = "[version=%(version)s] [kind=%(kind)s] [field=%(field)s]"
        try:
            logger.info("Try predict " + log_predict, predicts)
            predictor = PredictTimeSeries(storage_conf, predicts)
            predictor.run()
        except Exception as err:
            logger.exception("Failed predict " + log_predict, predicts)
            last_exception = err

    return last_exception


def main():
    args = make_arg_parser().parse_args()

    storage_conf = yaml.full_load(resource.find("/configs/storage.yaml"))

    metrics_conf = yaml.full_load(resource.find("/configs/metrics.yaml"))

    yt_proxy = os.getenv("YT_PROXY", "hahn.yt.yandex.net")
    yt_pool = os.getenv("YT_POOL", "crypta_graph")

    source = get_source(args)
    date = get_date(args, source)
    exp_stats_metrics = get_exp_stats_metrics(metrics_conf["exp_stats_metrics"])
    metrics = list(filter_metrics(args, list(metrics_conf["metrics"]) + list(exp_stats_metrics)))

    check_stable_metrics(metrics, args.history_dir)

    mc_procesor = MetricsProcessor(
        date=date, version=args.version, metrics=metrics, source=source, storage_conf=storage_conf,
        history_dir=args.history_dir, export_dir=args.export_dir, debug_dir=args.debug_dir
    )

    last_predict_error = None

    if True:
        # calc metrics and upload to YT
        result = list(mc_procesor.run(yt_proxy=yt_proxy, pool=yt_pool, is_embedded=False))
        mc_procesor.print_result(result)

        if args.local:
            # finish now
            return

        mc_procesor.upload_on_yt(result)

        if args.version in {V1_VERSION, V2_VERSION, V2_EXPERIMENT_VERSION}:
            # make predict
            last_predict_error = predict(args, {metrica["prefix"] for metrica in metrics}, metrics_conf, storage_conf)

    if True:
        upload_to_solomon(storage_conf["solomon"], data=mc_procesor.select_all())

    if mc_procesor.last_exception:
        raise mc_procesor.last_exception
    if last_predict_error:
        raise last_predict_error


if __name__ == "__main__":
    main()
