# coding: utf-8
from __future__ import unicode_literals
from __future__ import print_function

import datetime
import os
import yaml

from library.python import resource
from crypta.graph.metrics.telegram.report import (
    CommonReport,
    SimpleMoneyReport as MoneyReport,
    Greeting,
    JugglerTaskStatusReport,
    DutyReport,
)

import telebot


class TelegramBot(object):
    def __init__(self):
        self.bot = telebot.TeleBot(os.environ.get("TELEGRAM_TOKEN"))
        self.chats = os.environ.get("TELEGRAM_CHAT_IDS").split(",")

    def send_image(self, image_path, caption):
        for chat in self.chats:
            if image_path.startswith("/"):
                with open(image_path, "rb") as image:
                    self.bot.send_photo(chat, image, caption=caption)
            else:
                self.bot.send_photo(chat, image_path, caption=caption)

    def send_sticker(self, image_id):
        for chat in self.chats:
            self.bot.send_sticker(chat, image_id)

    def send_message(self, text, **kwargs):
        for chat in self.chats:
            self.bot.send_message(chat, text, **kwargs)


class Bot(object):
    """proxy for bot"""

    def __init__(self):
        self.tg = TelegramBot()

    def send_image(self, image_path, caption):
        self.tg.send_image(image_path, caption)

    def send_sticker(self, image_id):
        self.tg.send_sticker(image_id)

    def send_message(self, text, **kwargs):
        self.tg.send_message(text, **kwargs)


class PrintBot(object):
    """Just prints messages to console"""

    def send_image(self, image_path, caption):
        print(caption)
        print(image_path)

    def send_sticker(self, image_id):
        print(image_id)

    def send_message(self, text, **kwargs):
        print(text)


def main():
    config = yaml.full_load(resource.find("config.yaml"))
    bot = Bot()

    image = CommonReport(config).run()
    bot.send_image(image, caption="#matching_stat")

    image, sticker = MoneyReport(config).run()

    bot.send_sticker(sticker)
    bot.send_image(image, caption="#matching_stat")

    for message in Greeting(config).run():
        if message.image is not None:
            bot.send_image(message.image, caption="#greeting")
        if message.sticker is not None:
            bot.send_sticker(message.sticker)
        if message.text is not None:
            bot.send_message(message.text)

    for env in ("Prod", "Prestable"):
        for message in JugglerTaskStatusReport(env=env).run():
            bot.send_message(message, parse_mode="MarkdownV2")

    duty_report = DutyReport(
        tvm_id=int(os.environ.get("TVM_ID")),
        tvm_secret=os.environ.get("TVM_SECRET"),
        staff_table=config["tables"]["staff"],
    )
    for message in duty_report.run():
        bot.send_message(message, parse_mode="MarkdownV2")

    today = datetime.datetime.today()
    if today.weekday() < 5:
        quotelist = config[config["quotes"]]
        quote = quotelist[(today - datetime.datetime(today.year, 1, 1)).days % len(quotelist)]
        bot.send_message("STAND UP AND FIGHT!\n\n> {quote}".format(quote=quote))
