import logging

from cached_property import cached_property

import yt.wrapper as yt

from yt.wrapper.ypath import ypath_join
from crypta.lib.python.yql_runner.base_parser import BaseParser


def get_logger():
    logger = logging.getLogger("stats")
    logger.setLevel(logging.INFO)
    console = logging.StreamHandler()
    console.setLevel(logging.INFO)
    formatter = logging.Formatter("%(asctime)s [%(levelname)s] %(message)s")
    console.setFormatter(formatter)
    logger.addHandler(console)
    return logger


logger = get_logger()


class YtProcessor(object):
    def __init__(self, yt_path, schema):
        self.table_name = yt_path
        self.schema = schema

    def _get_table_state(self):
        return yt.get("{0}/@tablets/0/state".format(self.table_name))

    def prepare_table(self):
        if not yt.exists(self.table_name):
            logger.info("Create yt table if not exists [%s]", self.table_name)
            yt.create(
                "table",
                self.table_name,
                recursive=True,
                attributes={
                    "schema": self.schema,
                    "dynamic": True,
                    "optimize_for": "scan",
                    "enable_dynamic_store_read": True
                },
            )

        if self._get_table_state() == "unmounted":
            self.mount()

    def insert_data(self, data):
        data = data.full_dataframe.to_dict(orient="records")
        for row in data:
            row["is_approved_domain"] = bool(row["is_approved_domain"])

        logger.debug("Metrics data: %s", data)
        logger.info("Upload metrics data to [%s]", self.table_name)

        yt.insert_rows(self.table_name, data, update=True, format=yt.JsonFormat(encoding="utf-8", encode_utf8=False))

    def mount(self):
        logger.info("Mount dynamic table [%s]", self.table_name)
        yt.mount_table(self.table_name, sync=True)

    def unmount(self):
        logger.info("Unmount dynamic table [%s]", self.table_name)
        yt.unmount_table(self.table_name, sync=True)


class UserParamsStatsParser(BaseParser):

    BSWATCH_LOG_DIR = "//home/logfeller/logs/bs-watch-log/1d"
    QUERY_TEMPLATE = "main.sql.j2"

    def __init__(self, date, yt_proxy, pool, soup_cooked_edges=None, approved_domens_table=None, output_dir=None, is_embedded=False, **kwargs):
        super(UserParamsStatsParser, self).__init__(date, yt_proxy, pool, is_embedded=is_embedded, **kwargs)
        self.approved_domens_table = approved_domens_table or "//home/crypta/{env}/graph/config/wl_client_user_id/approved_domens".format(env=self.crypta_env)
        self.soup_cooked_edges = soup_cooked_edges or "//home/crypta/{env}/state/graph/v2/soup/cooked/soup_edges".format(env=self.crypta_env)
        self.output = self.get_output_metrics_path(output_dir)

    def get_context_data(self, **kwargs):
        context = super(UserParamsStatsParser, self).get_context_data(**kwargs)
        context.update(
            appmetrica_logs="//home/logfeller/logs/appmetrica-yandex-events/1d",
            appmetrica_external_logs="//home/logfeller/logs/appmetrica-external-events/1d",
            approved_domens_table=self.approved_domens_table,
            soup_cooked_edges=self.soup_cooked_edges,
            metrika_bs_watchlog=self.BSWATCH_LOG_DIR,
        )
        return context

    def get_output_metrics_path(self, output_dir=None):
        if not output_dir:
            output_dir = "//home/crypta/{env}/graph/metrics/user_params".format(env=self.crypta_env)
        return ypath_join(output_dir, "metrics")

    @cached_property
    def output_schema(self):
        return [
            {"name": "dt", "type": "string", "required": True, "sort_order": "ascending"},
            {"name": "id1Type", "type": "string", "required": True, "sort_order": "ascending"},
            {"name": "id2Type", "type": "string", "required": True, "sort_order": "ascending"},
            {"name": "logSource", "type": "string", "required": True, "sort_order": "ascending"},
            {"name": "sourceType", "type": "string", "required": True, "sort_order": "ascending"},
            {"name": "domain", "type": "string", "required": True, "sort_order": "ascending"},
            {"name": "is_approved_domain", "type": "boolean", "required": False},
            {"name": "counterids", "type": "any", "required": False},
            {"name": "found_id2_in_soup_ratio", "type": "double", "required": False},
            {"name": "count", "type": "uint64", "required": False},
        ]

    def run(self, **kwargs):
        logger.info("Running user params stats parser on [%s]", self.date)

        result = super(UserParamsStatsParser, self).run(**kwargs)

        logger.info("Finish counting of user params stats parser")

        yp = YtProcessor(self.output, self.output_schema)
        yp.prepare_table()
        yp.insert_data(result)
