-- input tables
$appmetrica_logs = [
    '{{ appmetrica_logs }}/' || $date,
    '{{ appmetrica_external_logs }}/' || $date,

];

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- for custom user attributes

{% include 'for_custom_user_attributes.sql.j2' %}

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

$preparsed_app_metrica = (
SELECT
    `UUID` AS `uuid`,
    APIKey AS api_key,
    AppID AS app_id,

    ProfileID AS profile_id, -- Custom profile id of user issued by the seller an application developer via SDK setUserProfileID()

    -- Custom Attributes https://appmetrica.yandex.ru/docs/data-collection/profile-attributes.html#custom
    IF(ListLength(CustomAttribute_Ids) > 0, CustomAttribute_Ids, ListCreate(Uint64)) AS custom_attribute_ids,
    IF(ListLength(CustomAttribute_Ids) > 0, CustomAttribute_Types, ListCreate(Uint32)) AS custom_attribute_types,
    IF(ListLength(CustomAttribute_Ids) > 0, CustomAttribute_StringValues, ListCreate(String)) AS custom_attribute_strings,
    IF(ListLength(CustomAttribute_Ids) > 0, CustomAttribute_NumberValues, ListCreate(Double)) AS custom_attribute_doubles,

FROM EACH($appmetrica_logs)
WHERE AppID != 'ru.yandex.botan'
    AND  EventType NOT IN {11, 4, 15, 31, 32}
        -- 'EVENT_STATBOX'
        -- 'EVENT_CLIENT'
        -- 'EVENT_%_NOTIFICATION'
    AND `UUID` IS NOT NULL
    AND (
        ProfileID IS NOT NULL
        OR ListLength(CustomAttribute_Ids) > 0
    )
);

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- Common action for counting metrics

DEFINE ACTION $parse_ids($prepared_log, $id2_type, $source_type, $is_approved) AS

    $ids_from_log =
    SELECT
        id2,
        app_id,
        is_approved,
        AGG_LIST_DISTINCT(api_key) AS api_keys,
    FROM (
        SELECT
            `uuid`,
            id2,
            app_id,
            CAST(api_key AS String) AS api_key,
            $is_approved(api_key) AS is_approved,
        FROM $prepared_log()
    )
    GROUP BY `uuid`, id2, app_id, is_approved;

    $ids_from_log_with_soup =
    SELECT
        log.app_id AS app_id,
        log.api_keys AS api_keys,
        log.is_approved AS is_approved,
        IF(soup.id2 IS NULL, FALSE, TRUE) AS found_id2_in_soup,
    FROM $ids_from_log AS log
    LEFT JOIN ANY $get_typed_soup_vertices($id2_type) AS soup
    USING(id2);

    INSERT INTO @metrics
    SELECT
        $date AS dt,
        IdType::UUID() AS id1Type,
        $id2_type() AS id2Type,
        LogSource::METRIKA_MOBILE_LOG() AS logSource,
        $source_type() AS sourceType,
        app_id ?? '' AS `domain`,
        MAX(is_approved) ?? FALSE AS is_approved_domain,
        SOME(api_keys) AS counterids,
        AVG(found_id2_in_soup) AS found_id2_in_soup_ratio,
        COUNT(*) AS count,
    FROM $ids_from_log_with_soup
    GROUP BY app_id;

END DEFINE;

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- Count metrics for profile_id
DEFINE SUBQUERY $prepare_profile_ids($is_valid, $normalize) AS
    SELECT
        `uuid`,
        $normalize(profile_id) AS id2,
        app_id,
        api_key,
    FROM $preparsed_app_metrica
    WHERE $is_valid(profile_id);
END DEFINE;

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- uuid-email
$always_approve = ($api_key) -> ( True );

DEFINE SUBQUERY $prepare_profile_ids_as_email() AS
    SELECT * FROM $prepare_profile_ids(Identifiers::IsValidEmail, Identifiers::NormalizeEmail);
END DEFINE;

DO $parse_ids(
    $prepare_profile_ids_as_email,
    IdType::EMAIL,
    SourceType::APP_METRICA_USER_PROFILE,
    $always_approve
);

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- uuid-phone

$phone_for_profileid_approved = ($api_key) -> ( $api_key IN  $PHONE_PROFILEID_APPROVED_APIKey );

DEFINE SUBQUERY $prepare_profile_ids_as_phone() AS
    SELECT * FROM $prepare_profile_ids($is_valid_phone, Identifiers::NormalizePhone);
END DEFINE;

DO $parse_ids(
    $prepare_profile_ids_as_phone,
    IdType::PHONE,
    SourceType::APP_METRICA_USER_PROFILE,
    $phone_for_profileid_approved
);

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- Count metrics for custom attributes

DEFINE SUBQUERY $prepare_custom_attribute_ids($get_ids) AS
    SELECT
        `uuid`,
        id2,
        app_id,
        api_key,
    FROM (
        SELECT
            $get_ids(custom_attribute_ids, custom_attribute_types, custom_attribute_strings, custom_attribute_doubles) AS ids2,
            `uuid`,
            app_id,
            api_key,
        FROM $preparsed_app_metrica
        WHERE ListLength(custom_attribute_ids) > 0
            AND ListLength($get_ids(custom_attribute_ids, custom_attribute_types, custom_attribute_strings, custom_attribute_doubles)) > 0
    ) FLATTEN LIST BY ids2 AS id2;
END DEFINE;

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- uuid-email
$filter_emails = ($ids, $strings) -> {
    $structs = ListMap(
        ListZipAll($ids, $strings),
        ($t) -> ( <|
           id: $t.0, -- UInt64
           value: $t.1, -- string
        |> )
    );
    RETURN ListFilter($structs, ($s) -> ( $s.id IN $EMAIL_CUSTOM_ATTRIBUTE_IDS AND Identifiers::IsValidEmail($s.value) ));
};

$get_emails = ($ids, $types, $strings, $numbers) -> {
    RETURN ListMap(
        $filter_emails($ids, $strings),
        ($s) -> ( Identifiers::NormalizeEmail($s.value) )
    );
};

$email_for_custom_attr_approved = ($api_key) -> ( $api_key IN  $EMAIL_CUSTOM_ATTRIBUTE_APPROVED_APIKey );

DEFINE SUBQUERY $prepare_custom_attribute_ids_as_email() AS
    SELECT * FROM $prepare_custom_attribute_ids($get_emails);
END DEFINE;

DO $parse_ids(
    $prepare_custom_attribute_ids_as_email,
    IdType::EMAIL,
    SourceType::APP_METRICA_CUSTOM_ATTRIBUTE,
    $email_for_custom_attr_approved
);

-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
-- uuid-phone
$filter_phones = ($ids, $types, $strings, $numbers) -> {
    $structs = ListMap(
        ListZipAll($ids, $types, $strings, $numbers),
        ($t) -> ( <|
           id: $t.0, -- UInt64
           value: IF($t.1 == 1, CAST($t.3 AS String), $t.2) -- string
           --  type = $t.1: 0=string 1=number 2=bool 3=counter 4=price
        |> )
    );
    RETURN ListFilter($structs, ($s) -> ( $s.id IN $PHONE_CUSTOM_ATTRIBUTE_IDS AND $is_valid_phone($s.value) ));
};

$get_phones = ($ids, $types, $strings, $numbers) -> {
    RETURN ListMap(
        $filter_phones($ids, $types, $strings, $numbers),
        ($s) -> ( Identifiers::NormalizePhone($s.value) )
    );
};

$phone_for_custom_attr_approved = ($api_key) -> ( $api_key IN  $PHONE_CUSTOM_ATTRIBUTE_APPROVED_APIKey );

DEFINE SUBQUERY $prepare_custom_attribute_ids_as_phone() AS
    SELECT * FROM $prepare_custom_attribute_ids($get_phones);
END DEFINE;

DO $parse_ids(
    $prepare_custom_attribute_ids_as_phone,
    IdType::PHONE,
    SourceType::APP_METRICA_CUSTOM_ATTRIBUTE,
    $phone_for_custom_attr_approved
);
