from luigi import (
    Task,
    Target,
    Parameter
)
import yt.wrapper as yt
from crypta.lib.python.yql.client import create_yql_client
from crypta.graph.mrcc.lib import (
    MRConnectedComponentsYT,
)
import crypta.graph.mrcc.bin.config as config
from utils import cached_property
import library.python.resource as rs

import logging


logger = logging.getLogger(__name__)


class Targets(object):

    class YtTarget(Target):

        def __init__(self, yt):
            self.yt = yt

    class PathYtTarget(YtTarget):

        def __init__(self, yt, path):
            Targets.YtTarget.__init__(self, yt)
            self.path = path

    class Exists(PathYtTarget):

        def exists(self):
            return self.yt.exists(self.path)

    class NotEmpty(PathYtTarget):

        def exists(self):
            _exists = self.yt.exists(self.path)
            if _exists:
                row_count = self.yt.get_attribute(self.path, 'row_count', 0)
                _not_empty = row_count > 0
                return _exists and _not_empty
            return _exists

    class TableIsActual(PathYtTarget):

        def __init__(self, yt, path, date):
            Targets.PathYtTarget.__init__(self, yt, path)
            self.date = date

        def exists(self):
            if self.yt.exists(self.path):
                table_attr = self.yt.get(self.path + '/@')
                date_str = table_attr.get('generate_date', '0001-01-01')
                return self.date <= date_str
            return False

    class TableHasAttribute(PathYtTarget):

        def __init__(self, yt, path, attr, value):
            Targets.PathYtTarget.__init__(self, yt, path)
            self.attr = attr
            self.value = value

        def exists(self):
            if not self.yt.exists(self.path):
                return False
            if not self.yt.has_attribute(self.path, self.attr):
                return False
            if self.value is None:
                return True
            actual_value = self.yt.get_attribute(self.path, self.attribute)
            return actual_value == self.value

    def __init__(self, yt):
        self.yt = yt

    def exists(self, table):
        return Targets.Exists(self.yt, table)

    def not_empty(self, table):
        return Targets.NotEmpty(self.yt, table)

    def table_is_actual(self, table, date):
        return Targets.TableIsActual(self.yt, table, date)

    def table_has_attribute(self, table, attr, value=None):
        return Targets.TableHasAttribute(table, attr, value)


class BaseTask(Task):

    @cached_property
    def yt(self):
        client = yt.YtClient(proxy=config.Yt.PROXY, token=config.Yt.TOKEN)
        client.targets = Targets(client)
        client.token = config.Yt.TOKEN
        client.path = yt.YPath
        client.config["pool"] = config.Yt.POOL
        client.pool = config.Yt.POOL
        client.proxy = client.config['proxy']['url']
        client.transaction_id = None
        return client

    @cached_property
    def yql(self):
        transaction = self.yt.transaction_id
        client = create_yql_client(
            yt_proxy=config.Yt.PROXY,
            token=config.Yql.TOKEN,
            transaction=transaction,
            db=config.Yql.DB,
            yql_server=config.Yql.SERVER,
            yql_port=config.Yql.PORT,
            pool=config.Yt.POOL,
        )
        return client

    @staticmethod
    def render_resource(name, **kwargs):
        return rs.find(name).format(**kwargs)

    def run(self):
        with self.yt.Transaction() as transaction:
            self.yt.transaction_id = str(transaction.transaction_id)
            logger.info("Running in transaction %s", self.yt.transaction_id)
            self._run()
            self.yt.transaction_id = None
        logger.info("Finishing task %s", self.__class__.__name__)


class Matching(BaseTask):

    edges_path = Parameter()
    destination = Parameter()

    def output(self):
        yield self.yt.targets.not_empty(self.destination)

    def get_normalize_query(self, edges, output):
        return rs.find("/yql/normalize.yql").format(edges=edges, output=output)

    @property
    def schema(self):
        return [{"name": "cryptaid", "type": "string", "required": True},
                {"name": "yandexuid", "type": "uint64", "required": True}]

    def _run(self):
        self.yt.create_table(self.destination,
                             attributes={"schema": self.schema,
                                         "compression_codec": "brotli_3"})
        mrcc_job = MRConnectedComponentsYT(self.yt)
        if not mrcc_job.find_connected_components(self.edges_path):
            logger.error("Not converged")
            raise RuntimeError("Not converged")
        query = self.get_normalize_query(self.edges_path, self.destination)
        # I don't want yql fake columns, so use explicit yt sort
        self.yql.execute(query)
        self.yt.run_sort(self.destination,
                         sort_by=["cryptaid", "yandexuid"])
