#pragma once

#include <crypta/lib/python/native_yt/cpp/registrar.h>
#include <crypta/lib/python/native_yt/cpp/proto.h>
#include <crypta/graph/mrcc/lib/proto/messages.pb.h>
#include <iostream>

using NYT::IMapper;
using NYT::IReducer;
using NYT::TTableReader;
using NYT::TTableWriter;

namespace NStarOperations {

    class TLargeStarMapper: public IMapper<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
    public:
        virtual void Do(TReader* input, TWriter* output) override {
            for (; input->IsValid(); input->Next()) {
                const auto& undirectedEdge = input->GetRow();
                const auto& source = undirectedEdge.GetSource();
                const auto& destination = undirectedEdge.GetDestination();
                TGraphEdge forward, backward;
                forward.SetSource(source);
                forward.SetDestination(destination);
                backward.SetSource(destination);
                backward.SetDestination(source);
                output->AddRow(forward);
                output->AddRow(backward);
            }
        }

        using TInputs = std::tuple<TGraphEdge>;
    };

    class TLargeStarReducer: public IReducer<TTableReader<TGraphEdge>, TTableWriter<::google::protobuf::Message>> {
    public:
        virtual void Do(TReader* input, TWriter* output) override {
            const auto& firstEdge = input->GetRow();
            const auto source = firstEdge.GetSource();
            const auto minVertex = Min(source, firstEdge.GetDestination());
            ui64 largeStarChanges = 0;
            for (; input->IsValid(); input->Next()) {
                const auto& edge = input->GetRow();
                const auto& destination = edge.GetDestination();
                if (destination > source) {
                    ++largeStarChanges;
                    TGraphEdge directedEdge;
                    directedEdge.SetSource(destination);
                    directedEdge.SetDestination(minVertex);
                    output->AddRow(directedEdge, 0);
                }
            }
            if (minVertex < source && largeStarChanges > 0) {
                TChangesCount changes;
                changes.SetCount(largeStarChanges);
                output->AddRow(changes, 1);
            }
        }

        using TOutputs = std::tuple<TGraphEdge, TChangesCount>;
};

    class TSmallStarMapper: public IMapper<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
    public:
        virtual void Do(TReader* input, TWriter* output) override {
            for (; input->IsValid(); input->Next()) {
                output->AddRow(input->GetRow());
            }
        }

        using TInputs = std::tuple<TGraphEdge>;
    };

    class TSmallStarReducer: public IReducer<TTableReader<TGraphEdge>, TTableWriter<::google::protobuf::Message>> {
    public:
        virtual void Do(TReader* input, TWriter* output) override {
            const auto& firstEdge = input->GetRow();
            const auto minDestination = firstEdge.GetDestination();
            ui64 smallStarChanges = 0;
            TGraphEdge reversedEdge;
            reversedEdge.SetSource(firstEdge.GetDestination());
            reversedEdge.SetDestination(firstEdge.GetSource());
            output->AddRow(reversedEdge, 0);
            input->Next();
            for (; input->IsValid(); input->Next()) {
                const auto& edge = input->GetRow();
                TGraphEdge outEdge;
                outEdge.SetSource(minDestination);
                outEdge.SetDestination(edge.GetDestination());
                output->AddRow(outEdge, 0);
                ++smallStarChanges;
            }
            if (smallStarChanges > 0) {
                TChangesCount changes;
                changes.SetCount(smallStarChanges);
                output->AddRow(changes, 1);
            }
        }

        using TOutputs = std::tuple<TGraphEdge, TChangesCount>;
    };
};

CYT_REGISTER_MAPREDUCER(NStarOperations::TLargeStarMapper, NStarOperations::TLargeStarReducer);
CYT_REGISTER_MAPREDUCER(NStarOperations::TSmallStarMapper, NStarOperations::TSmallStarReducer);
