#pragma once

#include <crypta/graph/mrcc_opt/proto/messages.pb.h>
#include <crypta/graph/mrcc_opt/lib/data.h>
#include <crypta/graph/mrcc_opt/lib/time.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/operation.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/vector.h>
#include <util/string/join.h>
#include <util/string/cast.h>
#include <iostream>

namespace NConnectedComponents {
    using NYT::IMapper;
    using NYT::IReducer;
    using NYT::TTableReader;
    using NYT::TTableWriter;
    static const ui64 MAX_RECORDS_PER_REDUCER = 1000;
    static const ui64 MAX_RECORDS_PER_SMALL_COMBINER = 350;

    namespace NChangesCountStats {
        const TString CHANGES_COUNT("ChangesCount");

        void WriteToCustomStatistics(ui64 changesCount) {
            if (changesCount > 0) {
                NYT::WriteCustomStatistics(
                    NYT::TNode()(CHANGES_COUNT, changesCount));
            }
        }

        ui64 ExtractFromOp(NYT::IOperationPtr op) {
            auto jobStatistics = op->GetJobStatistics();
            if (jobStatistics.HasCustomStatistics(CHANGES_COUNT)) {
                return static_cast<ui64>(*jobStatistics.GetCustomStatistics(CHANGES_COUNT).Sum());
            }
            return 0;
        }

        ui64 ExtractFromOp(NYT::IOperationPtr op, IOutputStream& logger) {
            auto changesCount = ExtractFromOp(op);
            logger << "\t" << CHANGES_COUNT << " " << changesCount << Endl;
            return changesCount;
        }
    }

    template <class TIdType>
    class TMROperations {
    public:
        using TGraphEdge = typename NConnectedComponents::TTypedGraphEdge<TIdType>::Type;
        using TEdges = typename NConnectedComponents::TTypedGraphEdge<TIdType>::Edges;

        class TExtractFirstRecordReducer: public IReducer<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                output->AddRow(input->MoveRow());
            }
        };

        class TExtractBigKeysCombiner: public IReducer<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                // reduce by source
                // sort by [source, destination]
                // source > destination
                auto edge = input->GetRow();
                bool big = false;
                for (ui64 count = 0; input->IsValid(); input->Next(), ++count) {
                    if (count >= MAX_RECORDS_PER_SMALL_COMBINER) {
                        big = true;
                        break;
                    }
                }
                if (big) {
                    edge.SetIsBig(true);
                    output->AddRow(edge);
                }
            }
        };

        class TExtractFirstForBigKeyReducer: public IReducer<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                // reduce by source
                // sort by [source, destination]
                // source > destination
                auto edge = input->GetRow();
                bool big = false;
                for (ui64 count = 0; input->IsValid(); input->Next(), ++count) {
                    auto row = input->MoveRow();
                    if (row.GetIsBig()) {
                        count += MAX_RECORDS_PER_SMALL_COMBINER;
                    }
                    if (count >= MAX_RECORDS_PER_SMALL_COMBINER * 2) {
                        big = true;
                        break;
                    }
                }
                if (big) {
                    edge.SetIsBig(true);
                    output->AddRow(edge);
                }
            }
        };

        class TMarkEdgesWithBigSourceMapper: public IMapper<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            TMarkEdgesWithBigSourceMapper() {
            }
            TMarkEdgesWithBigSourceMapper(const TString& edgesFile)
                : EdgesFile(edgesFile)
            {
            }
            Y_SAVELOAD_JOB(EdgesFile);

            void Start(TTableWriter<TGraphEdge>* /*writer*/) override {
                if (EdgesFile) {
                    TFileInput in(EdgesFile);
                    auto stringEdges = in.ReadAll();
                    TEdges edges;
                    Y_PROTOBUF_SUPPRESS_NODISCARD edges.ParseFromString(stringEdges);
                    for (const auto& edge : edges.GetEdges()) {
                        Vertices.insert(edge.GetSource());
                    }
                }
            }


            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                for (; input->IsValid(); input->Next()) {
                    auto edge = input->GetRow();
                    if (edge.GetSource() == edge.GetDestination()) {
                        continue;
                    }
                    bool isBig = IsBig(edge.GetSource());
                    edge.SetIsBig(isBig);
                    output->AddRow(edge);
                    if (isBig) {
                        output->AddRow(edge, 1);
                    }
                    if (IsBig(edge.GetDestination()) && edge.GetSource() < edge.GetDestination()) {
                        TGraphEdge out;
                        out.SetSource(edge.GetDestination());
                        out.SetDestination(edge.GetSource());
                        output->AddRow(out, 2);
                    }
                }
            }
        private:
            TString EdgesFile="";
            THashSet<TIdType> Vertices{};

            bool IsBig(const TIdType& id) {
                return Vertices.contains(id);
            }
        };


        class TLargeStarMapper: public IMapper<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            TLargeStarMapper() {
            }
            TLargeStarMapper(const TString& edgesFile)
                : EdgesFile(edgesFile)
            {
            }
            Y_SAVELOAD_JOB(EdgesFile);

            void Start(TTableWriter<TGraphEdge>* /*writer*/) override {
                if (EdgesFile) {
                    TFileInput in(EdgesFile);

                    auto stringEdges = in.ReadAll();
                    TEdges edges;
                    Y_PROTOBUF_SUPPRESS_NODISCARD edges.ParseFromString(stringEdges);
                    for (const auto& edge : edges.GetEdges()) {
                        Vertices.insert(edge.GetSource());
                    }
                }
            }


            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                for (; input->IsValid(); input->Next()) {
                    const auto& edge = input->GetRow();
                    const auto& source = edge.GetSource();
                    const auto& destination = edge.GetDestination();
                    if (source == destination) {
                        continue;
                    }
                    TGraphEdge forward, backward;
                    forward.SetSource(source);
                    forward.SetDestination(destination);
                    backward.SetSource(destination);
                    backward.SetDestination(source);
                    output->AddRow(backward);
                    if (edge.GetIsBig() || IsBig(source)) {
                        if (source > destination) {
                            ythrow yexception() << "Source must be less than destination for big sources, ("
                                                << source << ", " << destination << ")";
                        }
                    } else {
                        output->AddRow(forward);
                    }
                }
            }
        private:
            TString EdgesFile="";
            THashSet<TIdType> Vertices{};

            bool IsBig(const TIdType& id) {
                return Vertices.contains(id);
            }
        };

        class TLargeStarReducer: public IReducer<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                // reduce by source
                // sort by [source, destination]
                const auto& firstEdge = input->GetRow();
                const auto source = firstEdge.GetSource();
                auto minVertex = Min(source, firstEdge.GetDestination());
                ui64 changesCount = 0;
                auto prevDestination = firstEdge.GetDestination();
                for (ui64 rowsCount = 0; input->IsValid(); input->Next(), ++rowsCount) {
                    const auto& edge = input->GetRow();
                    auto destination = edge.GetDestination();
                    bool skip = (destination == prevDestination && rowsCount > 0);
                    if ((!skip) && destination > source) {
                        ++changesCount;
                        TGraphEdge out;
                        out.SetSource(destination);
                        out.SetDestination(minVertex);
                        out.SetMergeBy(source);
                        output->AddRow(out, 0);
                    }
                    if (rowsCount > MAX_RECORDS_PER_REDUCER) {
                        TGraphEdge stopEdge;
                        stopEdge.SetSource(destination);
                        stopEdge.SetDestination(minVertex);
                        stopEdge.SetMergeBy(source);
                        stopEdge.SetIsBig(true);
                        output->AddRow(stopEdge, 1);
                        break;
                    }
                    prevDestination = destination;
                    if (skip) {
                        continue;
                    }
                }
                if (minVertex == source) {
                    changesCount = 0;
                }
                NChangesCountStats::WriteToCustomStatistics(changesCount);
            }
        };

        class TLSPostProcessBigVerticesMapper: public IMapper<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            TLSPostProcessBigVerticesMapper() {
            }
            TLSPostProcessBigVerticesMapper(const TString& edgesFile)
                : EdgesFile(edgesFile)
            {
            }
            Y_SAVELOAD_JOB(EdgesFile);

            void Start(TTableWriter<TGraphEdge>* /*writer*/) override {
                TFileInput in(EdgesFile);
                auto stringEdges = in.ReadAll();
                TEdges edges;
                Y_PROTOBUF_SUPPRESS_NODISCARD edges.ParseFromString(stringEdges);
                for (const auto& edge : edges.GetEdges()) {
                    StopEdges[edge.GetMergeBy()] = edge;
                }
            }

            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                ui64 changesCount = 0;
                for (; input->IsValid(); input->Next()) {
                    const auto& row = input->GetRow();
                    const auto& source = row.GetSource();
                    if (StopEdges.contains(source)) {
                        const auto& stopEdge = StopEdges[source];
                        if (stopEdge.GetSource() < row.GetDestination()) {
                            TGraphEdge out;
                            out.SetSource(row.GetDestination());
                            out.SetDestination(stopEdge.GetDestination());
                            out.SetMergeBy(source);
                            output->AddRow(out);
                            if (out.GetDestination() != row.GetSource()) {
                                changesCount++;
                            }
                        }
                    }
                }
                NChangesCountStats::WriteToCustomStatistics(changesCount);
            }

        private:
            TString EdgesFile{};
            THashMap<TIdType, TGraphEdge> StopEdges{};
        };

        class TLSProcessBigVerticesMapper: public IMapper<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            TLSProcessBigVerticesMapper() {
            }
            TLSProcessBigVerticesMapper(const TString& edgesFile)
                : EdgesFile(edgesFile)
            {
            }
            Y_SAVELOAD_JOB(EdgesFile);

            void Start(TTableWriter<TGraphEdge>* /*writer*/) override {
                TFileInput in(EdgesFile);
                auto stringEdges = in.ReadAll();
                TEdges edges;
                Y_PROTOBUF_SUPPRESS_NODISCARD edges.ParseFromString(stringEdges);
                for (const auto& edge : edges.GetEdges()) {
                    Mins[edge.GetSource()] = edge.GetDestination();
                }
            }

            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                ui64 changesCount = 0;
                for (; input->IsValid(); input->Next()) {
                    const auto& row = input->GetRow();
                    TGraphEdge out;
                    out.SetSource(row.GetDestination());
                    out.SetDestination(GetDestination(row.GetSource()));
                    out.SetMergeBy(row.GetSource());
                    output->AddRow(out);
                    if (out.GetDestination() != row.GetSource()) {
                        ++changesCount;
                    }
                }
                NChangesCountStats::WriteToCustomStatistics(changesCount);
            }

        private:
            TString EdgesFile{};
            THashMap<TIdType, TIdType> Mins{};

            TIdType GetDestination(const TIdType& id) {
                if (Mins.contains(id)) {
                    return Mins[id];
                }
                return id;
            }
        };

        class TSmallStarReducer: public IReducer<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            TSmallStarReducer() {
            }
            TSmallStarReducer(const TString& edgesFile)
                : EdgesFile(edgesFile)
            {
            }
            Y_SAVELOAD_JOB(EdgesFile);
            void Start(TTableWriter<TGraphEdge>* /*writer*/) override {
                if (EdgesFile) {
                    TFileInput in(EdgesFile);
                    auto stringEdges = in.ReadAll();
                    TEdges edges;
                    Y_PROTOBUF_SUPPRESS_NODISCARD edges.ParseFromString(stringEdges);
                    for (const auto& edge : edges.GetEdges()) {
                        Vertices.insert(edge.GetDestination());
                    }
                }
            }
            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                // reduce by source
                // sort by [source, destination]
                // source > destination
                const auto& firstEdge = input->GetRow();
                const auto mergeBy = firstEdge.GetSource();
                const auto minDestination = firstEdge.GetDestination();
                int mainTableIndex = 0;
                if (IsBig(minDestination)) {
                    mainTableIndex = 1;
                }

                Yield(minDestination, mergeBy, mergeBy, mainTableIndex, output, true);

                ui64 changesCount = 0;
                auto prevDestination = minDestination;
                input->Next();
                for (; input->IsValid(); input->Next()) {
                    const auto& edge = input->GetRow();
                    if (edge.GetSource() < edge.GetDestination()) {
                        ythrow yexception() << "Destination must be less than source ("
                                            << edge.GetSource() << ", " << edge.GetDestination() << ") " << input->GetTableIndex();
                    }
                    if (edge.GetDestination() == prevDestination) {
                        continue;
                    }
                    prevDestination = edge.GetDestination();
                    ++changesCount;

                    Yield(minDestination, edge.GetDestination(), mergeBy, mainTableIndex, output);
                }
                NChangesCountStats::WriteToCustomStatistics(changesCount);
            }

        private:
            TString EdgesFile{};
            THashSet<TIdType> Vertices{};
            bool IsBig(const TIdType& id) {
                return Vertices.contains(id);
            }

            void Yield(const TIdType& source, const TIdType& destination, const TIdType& mergeBy, int mainTableIndex, TTableWriter<TGraphEdge>* output, bool firstEdge = false) {
                if (source == destination) {
                    return;
                }
                TGraphEdge edge;
                edge.SetSource(source);
                edge.SetDestination(destination);
                edge.SetMergeBy(mergeBy);
                output->AddRow(edge, mainTableIndex);

                if (mainTableIndex != 0 && (firstEdge || !IsBig(destination))) {
                    edge.SetIsBig(true);
                    output->AddRow(edge, 0);
                }
                if (IsBig(destination) && (source < destination)) {
                    TGraphEdge candidateToMin;
                    candidateToMin.SetSource(destination);
                    candidateToMin.SetDestination(source);
                    candidateToMin.SetMergeBy(mergeBy);
                    output->AddRow(candidateToMin, 2);
                }
            }
        };

        class TNaiveLargeStarMapper: public IMapper<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                for (; input->IsValid(); input->Next()) {
                    const auto& undirectedEdge = input->GetRow();
                    const auto& source = undirectedEdge.GetSource();
                    const auto& destination = undirectedEdge.GetDestination();
                    TGraphEdge forward, backward;
                    forward.SetSource(source);
                    forward.SetDestination(destination);
                    backward.SetSource(destination);
                    backward.SetDestination(source);
                    output->AddRow(forward);
                    output->AddRow(backward);
                }
            }
        };

        class TNaiveLargeStarReducer: public IReducer<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
        public:
            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                const auto& firstEdge = input->GetRow();
                const auto source = firstEdge.GetSource();
                const auto minVertex = Min(source, firstEdge.GetDestination());
                ui64 largeStarChanges = 0;
                for (; input->IsValid(); input->Next()) {
                    const auto& edge = input->MoveRow();
                    const auto& destination = edge.GetDestination();
                    if (destination > source) {
                        ++largeStarChanges;
                        TGraphEdge directedEdge;
                        directedEdge.SetSource(destination);
                        directedEdge.SetDestination(minVertex);
                        output->AddRow(directedEdge);
                    }
                }
                if (minVertex == source) {
                    largeStarChanges = 0;
                }

                NChangesCountStats::WriteToCustomStatistics(largeStarChanges);
            }
        };

        class TNaiveSmallStarReducer: public IReducer<TTableReader<TGraphEdge>, TTableWriter<TGraphEdge>> {
            virtual void Do(TTableReader<TGraphEdge>* input, TTableWriter<TGraphEdge>* output) override {
                const auto& firstEdge = input->GetRow();
                const auto minDestination = firstEdge.GetDestination();
                ui64 smallStarChanges = 0;
                TGraphEdge reversedEdge;
                reversedEdge.SetSource(firstEdge.GetDestination());
                reversedEdge.SetDestination(firstEdge.GetSource());
                output->AddRow(reversedEdge, 0);
                input->Next();
                for (; input->IsValid(); input->Next()) {
                    const auto& edge = input->MoveRow();
                    TGraphEdge outEdge;
                    outEdge.SetSource(minDestination);
                    outEdge.SetDestination(edge.GetDestination());
                    output->AddRow(outEdge);
                    ++smallStarChanges;
                }

                NChangesCountStats::WriteToCustomStatistics(smallStarChanges);
            }
        };
    };

    REGISTER_MAPPER(NConnectedComponents::TMROperations<TString>::TLargeStarMapper);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<TString>::TLargeStarReducer);
    REGISTER_MAPPER(NConnectedComponents::TMROperations<TString>::TLSPostProcessBigVerticesMapper);
    REGISTER_MAPPER(NConnectedComponents::TMROperations<TString>::TLSProcessBigVerticesMapper);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<TString>::TSmallStarReducer);
    REGISTER_MAPPER(NConnectedComponents::TMROperations<TString>::TNaiveLargeStarMapper);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<TString>::TNaiveLargeStarReducer);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<TString>::TNaiveSmallStarReducer);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<TString>::TExtractFirstRecordReducer);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<TString>::TExtractFirstForBigKeyReducer);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<TString>::TExtractBigKeysCombiner);
    REGISTER_MAPPER(NConnectedComponents::TMROperations<TString>::TMarkEdgesWithBigSourceMapper);

    REGISTER_MAPPER(NConnectedComponents::TMROperations<ui64>::TLargeStarMapper);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<ui64>::TLargeStarReducer);
    REGISTER_MAPPER(NConnectedComponents::TMROperations<ui64>::TLSPostProcessBigVerticesMapper);
    REGISTER_MAPPER(NConnectedComponents::TMROperations<ui64>::TLSProcessBigVerticesMapper);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<ui64>::TSmallStarReducer);
    REGISTER_MAPPER(NConnectedComponents::TMROperations<ui64>::TNaiveLargeStarMapper);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<ui64>::TNaiveLargeStarReducer);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<ui64>::TNaiveSmallStarReducer);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<ui64>::TExtractFirstRecordReducer);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<ui64>::TExtractFirstForBigKeyReducer);
    REGISTER_REDUCER(NConnectedComponents::TMROperations<ui64>::TExtractBigKeysCombiner);
    REGISTER_MAPPER(NConnectedComponents::TMROperations<ui64>::TMarkEdgesWithBigSourceMapper);

}
