#pragma once
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/operation.h>
#include <crypta/graph/mrcc_opt/lib/mrcc.h>
#include <crypta/graph/mrcc_opt/lib/yt.h>
#include <crypta/graph/mrcc_opt/lib/data.h>
#include <crypta/graph/mrcc_opt/proto/messages.pb.h>
#include <crypta/graph/mrcc_opt/soup/proto/messages.pb.h>
#include <iostream>


namespace NConnectedComponents {
namespace NSoup {
    template <class T> class TTypedSoupVertex {};

    template <> class TTypedSoupVertex<TString> {
        public:
            typedef TStringSoupVertex Type;
    };

    template <> class TTypedSoupVertex<ui64> {
        public:
            typedef TUintSoupVertex Type;
    };

    template <class TIdType>
    class TMROperations {
    public:
        using TGraphEdge = typename TTypedGraphEdge<TIdType>::Type;
        using TEdges = typename TTypedGraphEdge<TIdType>::Edges;
        using TSoupVertex = typename NConnectedComponents::NSoup::TTypedSoupVertex<TIdType>::Type;


        class TExtractFirstRecord: public IReducer<TTableReader<TSoupVertex>, TTableWriter<TSoupVertex>> {
        public:
            virtual void Do(TTableReader<TSoupVertex>* input, TTableWriter<TSoupVertex>* output) override {
                output->AddRow(input->MoveRow());
            }
        };

        class TPrepareEdgesMapper: public IMapper<TTableReader<TSoupEdge>, TTableWriter<TGraphEdge>> {
        public:
            virtual void Do(TTableReader<TSoupEdge>* input, TTableWriter<TGraphEdge>* output) override {
                for (; input->IsValid(); input->Next()) {
                    const auto& row = input->GetRow();

                    auto source = NConnectedComponents::GetID<TIdType>({row.GetID1(), row.GetID1Type()});
                    auto destination = NConnectedComponents::GetID<TIdType>({row.GetID2(), row.GetID2Type()});
                    auto out = NConnectedComponents::PrepareOutputEdge(source, destination);
                    if (source == destination) {
                        continue;
                    }

                    output->AddRow(out);
                }
            }
        };

        class TConvertVerticesToEdges: public IMapper<TTableReader<TSoupVertex>, TTableWriter<TGraphEdge>> {
        public:
            virtual void Do( TTableReader<TSoupVertex>* input, TTableWriter<TGraphEdge>* output) override {
                for (; input->IsValid(); input->Next()) {
                    const auto& row = input->GetRow();

                    auto source = row.GetComponentID();
                    auto destination = NConnectedComponents::GetID<TIdType>({row.GetID(), row.GetIDType()});
                    auto out = NConnectedComponents::PrepareOutputEdge(source, destination);
                    if (source == destination) {
                        continue;
                    }

                    output->AddRow(out);
                }
            }
        };

        class TExtractVerticesMapper: public IMapper<TTableReader<::google::protobuf::Message>, TTableWriter<TSoupVertex>> {
        public:
            virtual void Do(NYT::TTableReader<google::protobuf::Message>* input, TTableWriter<TSoupVertex>* output) override {
                for (; input->IsValid(); input->Next()) {
                    if (input->GetTableIndex() == 0) {
                        const auto& row = input->GetRow<TSoupEdge>();
                        output->AddRow(GetOutputRecord(row.GetID1(), row.GetID1Type()));
                        output->AddRow(GetOutputRecord(row.GetID2(), row.GetID2Type()));
                    } else {
                        const auto& row = input->GetRow<TSoupVertex>();
                        output->AddRow(GetOutputRecord(row.GetID(), row.GetIDType()));
                    }
                }
            }
        private:
            TSoupVertex GetOutputRecord(const TString& id, const TString& idType) {
                TSoupVertex out;
                out.SetID(id);
                out.SetIDType(idType);
                auto vertexID = NConnectedComponents::GetID<TIdType>({id, idType});
                out.SetDestination(vertexID);
                return out;
            }
        };

        class TMatchSoupVerticesWithFinalEdges {
        public:

            class TCombiner: public IReducer<TTableReader<TSoupVertex>, TTableWriter<TSoupVertex>> {
                public:
                virtual void Do(TTableReader<TSoupVertex>* input, TTableWriter<TSoupVertex>* output) override {
                    output->AddRow(input->MoveRow());
                    input->Next();
                    if (input->IsValid()) {
                        output->AddRow(input->MoveRow());
                    }
                }
            };

            class TReducer: public IReducer<TTableReader<TSoupVertex>, TTableWriter<TSoupVertex>> {
            public:
                virtual void Do(TTableReader<TSoupVertex>* input, TTableWriter<TSoupVertex>* output) override {
                    //  reduce by DESTINATION
                    //  sort by DESTINATION, SOURCE
                    const auto& row = input->GetRow();
                    if (row.GetSource()) {
                        return;
                    }
                    TSoupVertex out;
                    out.SetID(row.GetID());
                    out.SetIDType(row.GetIDType());
                    out.SetComponentID(row.GetDestination());
                    for (ui64 i = 0; input->IsValid(); input->Next(), ++i) {
                        const auto& row = input->GetRow();
                        if (row.GetSource()) {
                            out.SetComponentID(row.GetSource());
                            break;
                        }
                        if (i > 1) {
                            throw yexception() << "Too much vertices for one hash";
                        }
                    }
                    output->AddRow(out);
                }
            };

        };
    };

    template <class TIdType>
    void ConvertDataToMRCCEdges(const TYT& yt, const TString& source, const TString& destination, IOutputStream& logger = Cout) {
        TMeasure measure(logger, __func__);
        NYT::TMapOperationSpec spec;
        spec.AddInput<TSoupEdge>(source);
        spec.template AddOutput<typename TMROperations<TIdType>::TGraphEdge>(destination);

        auto op = yt.Client->Map(
            spec,
            new typename TMROperations<TIdType>::TPrepareEdgesMapper,
            yt.CommonOperationOptions
        );

    }

    template <class TIdType>
    void ConvertComponentsToMRCCEdges(const TYT& yt, const TString& source, const TString& destination, IOutputStream& logger = Cout, bool append = false) {
        TMeasure measure(logger, __func__);
        NYT::TMapOperationSpec spec;
        spec.AddInput<typename TMROperations<TIdType>::TSoupVertex>(source);
        spec.template AddOutput<typename TMROperations<TIdType>::TGraphEdge>(NYT::TRichYPath(destination).Append(append));

        auto op = yt.Client->Map(
            spec,
            new typename TMROperations<TIdType>::TConvertVerticesToEdges,
            yt.CommonOperationOptions
        );

    }

    template <class TIdType>
    NYT::IOperationPtr ExtractVertices(const TYT& yt, const TString& source, const TString& destination, const TString& previousLabels,
                        IOutputStream& logger = Cout, bool wait = true) {

            TMeasure measure(logger, __func__);
            NYT::TMapReduceOperationSpec spec;
            spec
                .AddInput<TSoupEdge>(source)
                .template AddOutput<typename TMROperations<TIdType>::TSoupVertex>(destination)
                .ReduceBy({NGraphEdgeFields::DESTINATION})
                .SortBy({NGraphEdgeFields::DESTINATION});

            if (previousLabels) {
                spec.AddInput<typename TMROperations<TIdType>::TSoupVertex>(previousLabels);
            }
            NYT::TOperationOptions options(yt.CommonOperationOptions);
            options.Wait(wait);

            auto op = yt.Client->MapReduce(
                spec,
                new typename TMROperations<TIdType>::TExtractVerticesMapper,
                new typename TMROperations<TIdType>::TExtractFirstRecord,
                new typename TMROperations<TIdType>::TExtractFirstRecord,
                options
            );
            return op;
    }

    template <class TIdType>
    void ConvertMRCCEdgesToComponent(const TYT& yt, const TString& vertices, const TVector<TString>& sources, const TString& destination, IOutputStream& logger = Cout) {
            TMeasure measure(logger, __func__);
            NYT::TMapReduceOperationSpec spec;
            spec.template AddInput<typename TMROperations<TIdType>::TSoupVertex>(vertices);
            for (const auto& source: sources) {
                spec.template AddInput<typename TMROperations<TIdType>::TSoupVertex>(source);
            }
            spec.template AddOutput<typename TMROperations<TIdType>::TSoupVertex>(destination);
            spec.ReduceBy({NGraphEdgeFields::DESTINATION})
                .SortBy({NGraphEdgeFields::DESTINATION, NGraphEdgeFields::SOURCE});

            auto op = yt.Client->MapReduce(
                spec,
                nullptr,
                new typename TMROperations<TIdType>::TMatchSoupVerticesWithFinalEdges::TCombiner,
                new typename TMROperations<TIdType>::TMatchSoupVerticesWithFinalEdges::TReducer,
                yt.CommonOperationOptions
            );
    }
}

REGISTER_MAPPER(NSoup::TMROperations<TString>::TPrepareEdgesMapper);
REGISTER_MAPPER(NSoup::TMROperations<TString>::TConvertVerticesToEdges);
REGISTER_MAPPER(NSoup::TMROperations<TString>::TExtractVerticesMapper);
REGISTER_REDUCER(NSoup::TMROperations<TString>::TMatchSoupVerticesWithFinalEdges::TCombiner);
REGISTER_REDUCER(NSoup::TMROperations<TString>::TMatchSoupVerticesWithFinalEdges::TReducer);
REGISTER_REDUCER(NSoup::TMROperations<TString>::TExtractFirstRecord);


REGISTER_MAPPER(NSoup::TMROperations<ui64>::TPrepareEdgesMapper);
REGISTER_MAPPER(NSoup::TMROperations<ui64>::TConvertVerticesToEdges);
REGISTER_MAPPER(NSoup::TMROperations<ui64>::TExtractVerticesMapper);
REGISTER_REDUCER(NSoup::TMROperations<ui64>::TMatchSoupVerticesWithFinalEdges::TCombiner);
REGISTER_REDUCER(NSoup::TMROperations<ui64>::TMatchSoupVerticesWithFinalEdges::TReducer);
REGISTER_REDUCER(NSoup::TMROperations<ui64>::TExtractFirstRecord);

}

namespace NConnectedComponents {

    struct TSoupView {
        TSoupView() {}
    };

    template <class TIdType>
    struct TConverter<TSoupView, TIdType> {
        void ConvertDataToMRCCEdges(const TYT& yt, const TString& source, const TString& destination, const TSoupView&, IOutputStream& logger = Cout) {
            NConnectedComponents::NSoup::ConvertDataToMRCCEdges<TIdType>(yt.Client, source, destination, logger);
        }

        NYT::IOperationPtr ExtractVerticesFromData(const TYT& yt, const TString& source, const TString& destination, const TString& previousLabels, const TSoupView&, IOutputStream& logger = Cout, bool wait = true) {
            return NConnectedComponents::NSoup::ExtractVertices<TIdType>(yt.Client, source, destination, previousLabels, logger, wait);
        }

        void ConvertComponentsToMRCCEdges(const TYT& yt, const TString& source, const TString& destination, const TSoupView&, IOutputStream& logger, bool append) {
            NConnectedComponents::NSoup::ConvertComponentsToMRCCEdges<TIdType>(yt.Client, source, destination, logger, append);

        }

        void ConvertMRCCEdgesToComponents(const TYT& yt, const TString& vertices, const TVector<TString>& sources, const TString& destination, const TSoupView&, IOutputStream& logger) {
            NConnectedComponents::NSoup::ConvertMRCCEdgesToComponent<TIdType>(yt.Client, vertices, sources, destination, logger);
        }
    };
}


namespace NConnectedComponents {
    namespace NSoup {

        namespace NPaths {
            static const TString MAIN_SOUP_TABLE("//home/crypta/production/state/graph/v2/soup/cooked/soup_edges");
            static const TString SOUP_ADDITION_BY_DAY("//home/crypta/production/state/graph/v2/soup/day");
            static const TString MAIN_WORKDIR("//home/crypta/production/state/graph/v2/soup_cc/workdir");
            static const TString MAIN_COMPONENTS_TABLE("//home/crypta/production/state/graph/v2/soup_cc/components");
        }
        namespace NAttributes {
            static const TString GENERATE_DATE("@generate_date");
        }

        bool CollectEdgesFromDir(const TYT& yt, const TString& dirPath, const TString& dstPath) {
            if (!yt.Client->Exists(dirPath)) {
               return false;

            }
            NYT::TMergeOperationSpec spec;
            for (const auto& node: yt.Client->List(dirPath, NYT::TListOptions().AttributeFilter(NYT::TAttributeFilter().AddAttribute("type")))) {
                if (node.GetAttributes().At("type").AsString() == "table") {
                    auto source = NYT::TRichYPath(NYT::JoinYPaths(dirPath, node.AsString())).Columns({"id1", "id1Type", "id2", "id2Type"});
                    spec.AddInput(source);
                }
            }
            spec.Output(dstPath);

            yt.Client->Merge(spec, yt.GetCommonOperationOptionsWithJoinedSpec(NYT::TNode()("schema_inference_mode", "from_output")));
            return true;
        }

        TString GetPathToDateAttribute(const TString& path) {
            return NYT::JoinYPaths(path, NAttributes::GENERATE_DATE);
        }

        TString GetGenerateDateFromAttribute(NYT::IClientBasePtr client, const TString& path) {
            return client->Get(GetPathToDateAttribute(path)).AsString();
        }

        void SetGenerateDateAttribute(NYT::IClientBasePtr client, const TString& path, const TString& date) {
            client->Set(GetPathToDateAttribute(path), date);
        }

        bool RecountComponents(NYT::IClientBasePtr client,
                const TString& source = NPaths::MAIN_SOUP_TABLE,
                const TString& destination = NPaths::MAIN_COMPONENTS_TABLE,
                const TString& previousLabels = NPaths::MAIN_COMPONENTS_TABLE) {
            TString workdir(NPaths::MAIN_WORKDIR);

            NConnectedComponents::TSoupView soupView;
            NConnectedComponents::TDataPaths dataPaths(soupView, source, destination, workdir, previousLabels);
            if (previousLabels && !client->Exists(previousLabels)) {
                dataPaths.PreviousLabels = "";
            }
            auto mrcc = NConnectedComponents::TOptimizedStars<NConnectedComponents::TSoupView, ui64>(client, dataPaths, Cout);
            auto result = mrcc.Run();
            if (result) {
                SetGenerateDateAttribute(client, destination, GetGenerateDateFromAttribute(client, source));
            }
            return result;
        }

        bool AddEdgesToComponentsFromDate(NYT::IClientBasePtr client, const TString& date) {
            TString dateDir = NYT::JoinYPaths(NPaths::SOUP_ADDITION_BY_DAY, date);
            TYT yt(client);
            TString sourceData = NYT::JoinYPaths(NPaths::MAIN_WORKDIR, "day_addition");
            if (!CollectEdgesFromDir(yt, dateDir, sourceData)) {
                return false;
            }
            SetGenerateDateAttribute(client, sourceData, date);
            return RecountComponents(client, sourceData);
        }

    }
}
