#include "http_helper.h"
#include "exp_stats_requester.h"
#include "request_handler.h"
#include "request_logger.h"

#include <util/system/env.h>
#include <util/system/hostname.h>

namespace {
    using namespace NCryptaSolomonMetrics;

    namespace NProxyMetrics {
        const TString REPLY_OK{"reply.ok"};
        const TString REPLY_ERROR{"reply.error"};
        const TString INFLY{"infly"};
        const TString EXPSTATS_TIME{"expstats_time"};
    }

    const TMetricsInit InitMetrics = {
        {NProxyMetrics::REPLY_OK, EMetricType::COUNTER, {}},
        {NProxyMetrics::REPLY_ERROR, EMetricType::COUNTER, {}},
        {NProxyMetrics::INFLY, EMetricType::IGAUGE, {}},
        {NProxyMetrics::EXPSTATS_TIME, EMetricType::HISTOGRAM, {}, NMonitoring::TBucketBounds{5, 10, 20, 30, 40, 50, 60, 120, 240}},
    };

    TString SelectHostname() {
        const static TString DockerHostnameVar{"DEPLOY_POD_PERSISTENT_FQDN"};
        return GetEnv(DockerHostnameVar, FQDNHostName());
    }

    class TInflyHelper {
    public:
        TInflyHelper(NCryptaSolomonMetrics::TMetricsHelper& metrics)
        : Metrics(metrics) {
            Metrics.IncIGauge(NProxyMetrics::INFLY);
        }
        ~TInflyHelper() {
            Metrics.DecIGauge(NProxyMetrics::INFLY);
        }

    private:
        NCryptaSolomonMetrics::TMetricsHelper& Metrics;
    };
}

namespace NCrypta::NProxyExpStats {
    void TRequestHandler::ProxyExpStats(const ::NNeh::IRequestRef& r) {
        const static TString queryParamName{"query"};
        const static TString serializeParamName{"serialize"};

        TRequestLogger logMessage(r);
        auto* httpReq = dynamic_cast<NNeh::IHttpRequest*>(r.Get());
        NNeh::TDataSaver responseData;

        auto query = GetCgiParam(httpReq->Cgi(), queryParamName);
        auto serialize = GetCgiParam(httpReq->Cgi(), serializeParamName);
        TString reply;
        try {
            reply = DoExpStatsRequest(query, serialize);
        } catch (const yexception& e) {
            logMessage.SetCode("500").Write(e.what());
            Metrics.CounterIncrement(NProxyMetrics::REPLY_ERROR);
            httpReq->SendError(NNeh::IRequest::InternalError, e.what());
            return;
        }

        TString headers = "\r\nContent-Type: application/json";
        responseData.DoWrite(reply.data(), reply.size());
        httpReq->SendReply(responseData, headers);

        Metrics.HistogramRecord(NProxyMetrics::EXPSTATS_TIME, logMessage.GetDuration().Seconds());
        Metrics.CounterIncrement(NProxyMetrics::REPLY_OK);
    }

    void TRequestHandler::Solomon(const ::NNeh::IRequestRef& r) {
        auto reply = Metrics.Solomonify();
        NNeh::TDataSaver dataSaver;

        dataSaver.DoWrite(reply.data(), reply.size());
        r->SendReply(dataSaver);
    }

    void TRequestHandler::Pong(const ::NNeh::IRequestRef& r) {
        TRequestLogger logMessage(r);
        NNeh::TDataSaver dataSaver;
        TStringBuilder reply;

        reply << "OK "
              << Hostname;

        logMessage.Write(reply);
        dataSaver.DoWrite(reply.data(), reply.size());
        r->SendReply(dataSaver);
    }

    void TRequestHandler::ServeRequest(const ::NNeh::IRequestRef& r) {
        TInflyHelper infly(Metrics);

        if (auto it = RouteMap.find(r->Service()); it != RouteMap.end()) {
            it->second(r);
        } else {
            TRequestLogger logMessage(r, "404");
            r->SendError(NNeh::IRequest::NotExistService);
        }
    }

    TRequestHandler::TRequestHandler(NNeh::IServicesRef& server, const TString& listenAt)
    : Hostname(SelectHostname())
    , Metrics({}, InitMetrics)
    {
        RouteMap.insert({"ok", [this](const NNeh::IRequestRef& r) { this->Pong(r);}});
        RouteMap.insert({"solomon", [this](const NNeh::IRequestRef& r) { this->Solomon(r);}});
        RouteMap.insert({"exp_stats", [this](const NNeh::IRequestRef& r) { this->ProxyExpStats(r);}});

        for (const auto& [path, unused] : RouteMap) {
            server->Add(listenAt + "/" + path, *this);
        }
    }
}
