import logging
from os import path

from crypta.graph.soup.config.python import ID_TYPE

logger = logging.getLogger(__name__)


def get_types():
    return [id_type.Name for id_type in ID_TYPE.values()]


class MatchingPublicLinks(object):
    def __init__(self, yt_client, acl, id_types=None):
        self.yt = yt_client
        self.acl = acl
        if id_types is None:
            id_types = get_types()
        self.id_types = id_types

    def _ensure_dir(self, path, force=False, directory_acl=None):
        logger.info("Ensuring dir: %s", path)
        if force:
            self.yt.create(
                "map_node",
                path,
                recursive=True,
                force=True,
                attributes={"acl": directory_acl or self.acl.get_directory_acl()},
            )
        else:
            self.yt.create("map_node", path, recursive=True, ignore_existing=True)

    def _create_link(self, from_path, link_path):
        logger.info("Creating link %s -> %s", from_path, link_path)
        self.yt.link(from_path, link_path, force=True)

    def _create_copy(self, from_path, copy_path):
        logger.info("Creating copy %s -> %s", from_path, copy_path)
        self.yt.copy(from_path, copy_path, force=True)

    def publish_matching_links(self, matching_by_id_dir, public_by_id_dir):
        for id_type1 in self.id_types:
            ensure_public_path = path.join(public_by_id_dir, id_type1, "direct")
            self._ensure_dir(ensure_public_path)

            for id_type2 in self.id_types:
                if id_type1 == ID_TYPE.CRYPTA_ID.Name or id_type2 == ID_TYPE.CRYPTA_ID.Name:
                    from_path = path.join(matching_by_id_dir, id_type1, id_type2)
                    link_path = path.join(public_by_id_dir, id_type1, id_type2)
                else:
                    from_path = path.join(matching_by_id_dir, id_type1, "direct", id_type2)
                    link_path = path.join(public_by_id_dir, id_type1, "direct", id_type2)

                if self.yt.exists(from_path):
                    self._create_link(from_path, link_path)

    def publish_soup_links(self, soup_dir, public_by_id_dir, reinitialize=False):
        soup_tables = [t for t in self.yt.list(soup_dir) if self.yt.get_type(path.join(soup_dir, t)) == "table"]
        self._ensure_dir(public_by_id_dir, force=reinitialize)

        for id_type1 in self.id_types:
            id_type_dir = path.join(public_by_id_dir, id_type1)
            self._ensure_dir(id_type_dir, force=reinitialize)

            for id_type2 in self.id_types:
                if id_type2 == ID_TYPE.CRYPTA_ID.Name:
                    continue

                public_soup_dir = path.join(id_type_dir, "soup", id_type2)
                self.yt.create(
                    "map_node",
                    public_soup_dir,
                    force=True,
                    recursive=True,
                    attributes={
                        "acl": self.acl.get_pair_acl(id_type1, id_type2),
                        "inherit_acl": False,
                        "account": "crypta-graph",  # TODO remove me when possible
                    },
                )

                for t in soup_tables:
                    direct_types_prefix = id_type1 + "_" + id_type2 + "_"
                    reversed_types_prefix = id_type2 + "_" + id_type1 + "_"

                    if t.startswith(direct_types_prefix) or t.startswith(reversed_types_prefix):
                        soup_path = path.join(soup_dir, t)

                        link_name = t.replace(direct_types_prefix, "").replace(reversed_types_prefix, "")
                        link_path = path.join(public_soup_dir, link_name)

                        self._create_copy(soup_path, link_path)

    def publish_staff_links(self, staff_dir, public_staff_dir, reinitialize=False):
        self.yt.create(
            "map_node",
            public_staff_dir,
            force=True,
            recursive=True,
            attributes={
                "acl": self.acl.get_pair_acl("crypta_id", "staff"),
                "inherit_acl": False,
                "account": "crypta-graph",  # TODO remove me when possible
            },
        )
        self._create_copy(path.join(staff_dir, "yandexuid_ext"), path.join(public_staff_dir, "raw"))

    def publish_households_links(self, household_dir, public_household_dir, reinitialize=False):
        """ Link households tables into public directory """
        if not self.yt.exists(household_dir):
            return
        tables = [
            table
            for table in self.yt.list(household_dir)
            if self.yt.get_type(path.join(household_dir, table)) == "table"
        ]
        self._ensure_dir(public_household_dir, force=reinitialize, directory_acl=self.acl.get_basic_acl())
        # self.yt.set('{path}/@acl'.format(path=household_dir), self.acl.get_basic_acl())
        for table in tables:
            private_path = path.join(household_dir, table)
            public_path = path.join(public_household_dir, table)
            self._create_link(private_path, public_path)
