#include <crypta/graph/rt/brusilov/bin/config.pb.h>
#include <crypta/graph/rt/brusilov/processors/vulture.h>
#include <crypta/graph/rt/services/services.h>
#include <crypta/graph/rt/lib/sensors/build_version.h>

#include <ads/bsyeti/big_rt/lib/consuming_system/consuming_system.h>
#include <ads/bsyeti/big_rt/lib/utility/logging/logging.h>
#include <ads/bsyeti/big_rt/lib/deprecated/sensors/sensors.h>
#include <ads/bsyeti/big_rt/lib/deprecated/services/services.h>
#include <ads/bsyeti/big_rt/lib/supplier/supplier.h>

#include <ads/bsyeti/libs/profiling/solomon/exporter.h>

#include <ads/bsyeti/libs/ytex/client/cache.h>
#include <ads/bsyeti/libs/ytex/client/proto/config.pb.h>
#include <ads/bsyeti/libs/ytex/common/await.h>
#include <ads/bsyeti/libs/ytex/http/server.h>
#include <ads/bsyeti/libs/ytex/http/std_handlers.h>
#include <ads/bsyeti/libs/ytex/program/program.h>

#include <yt/yt/core/misc/shutdown.h>
#include <yt/yt/core/net/address.h>
#include <yt/yt/core/net/config.h>

#include <util/generic/xrange.h>
#include <util/string/join.h>
#include <util/thread/pool.h>


namespace NBrusilov {
    using namespace NBigRT;

    TConsumingSystemPtr CreateVultureSystem(NSFStats::TSolomonContext rootCtx,
                                            const TBrusilovConfig& config,
                                            const TBrusilovConfig::TProcessingSystem& systemConfig) {
        INFO_LOG << "Create processing system " << systemConfig.GetName() << "\n";

        NSFStats::TSolomonContext soloCtx{rootCtx, {{"system", systemConfig.GetName()}}};

        auto ytClients{NYTEx::NRpc::CreateClientsCache(config.GetYtRpc())};
        auto inflightLimiter{CreateProfiledInflightLimiter(soloCtx, systemConfig.GetMaxInflightBytes())};
        auto stateConfig{systemConfig.GetStateProcessorConfig()};

        Y_ENSURE(stateConfig.GetCodec() == "zlib-6"); // BB vulture now support only zlib-6 codec

        auto vultureStateManagerFactory{
            NYT::New<NRtCrypta::TVultureSMFactory>(
                systemConfig.GetStateManagerConfig(),
                stateConfig.GetDebounceCfg(),
                NRtCrypta::TVultureStateTableOperator({
                    .Table = stateConfig.GetTable(),
                    .Codec = stateConfig.GetCodec(),
                }),
                soloCtx
            )
        };

        auto consSystem{
            CreateConsumingSystem({
                .Config = systemConfig.GetConsumingSystem(),
                .SuppliersProvider = CreateSupplierFactoriesProvider({.ConfigsRepeated = systemConfig.GetSuppliers()}),
                .YtClients = ytClients,
                .ShardsProcessor = [=](TConsumingSystem::IConsumer& consumer) {
                    TVultureProcessor{
                        TVultureProcessor::TConstructionArgs{
                            consumer,
                            systemConfig.GetStatefulShardProcessorConfig(),
                            vultureStateManagerFactory,
                            NYTEx::CreateTransactionKeeper(ytClients, TDuration::Zero()),
                            inflightLimiter,
                            soloCtx,
                            NYT::GetCurrentInvoker()},
                        stateConfig}
                        .Run();
                },
                .SensorsContext = soloCtx,
            })
        };

        return consSystem;
    }

    void RunBrusilovService(const TBrusilovConfig& config, const NYT::TCancelableContextPtr& cancelableContext) {
        auto solomonExporter = NBSYeti::NProfiling::CreateSolomonExporter(config.GetSolomonExporter());
        solomonExporter->Start();

        auto httpServer = NYTEx::NHttp::CreateServer(config.GetHttpServer());
        NYTEx::NHttp::AddStandardHandlers(
            httpServer, cancelableContext, config,
            BIND([solomonExporter] {
                return NYT::TSharedRef::FromString(NBSYeti::NProfiling::GetSensors(solomonExporter));
            }));
        httpServer->Start();
        INFO_LOG << "Started http server" << Endl;

        auto rootCtx = NBigRT::MakeSolomonContext({});
        NCrypta::BuildVersionSensors(rootCtx);

        TVector<TConsumingSystemPtr> consSystems;

        for (const auto& systemConfig : config.GetProcessingSystems()) {
            consSystems.push_back(CreateVultureSystem(rootCtx, config, systemConfig));
        }

        for (auto consSystem : consSystems) {
            consSystem->Run();
        }

        NYTEx::WaitFor(cancelableContext);
        INFO_LOG << "Stopping consuming system\n";

        for (auto consSystem : consSystems) {
            NYT::NConcurrency::WaitUntilSet(consSystem->Stop());
        }
    }


    class TBrusilovProgram: public NYTEx::TProgram<TBrusilovConfig> {
    public:
        TBrusilovProgram()
            : NYTEx::TProgram<TBrusilovConfig>("brusilov") {
        }

        int DoRun(const NYT::TCancelableContextPtr& context) override {
            RunBrusilovService(Config(), context);
            return 0;
        }
    };
}

int main(int argc, const char** argv) {
    return NYTEx::RunProgram<NBrusilov::TBrusilovProgram>(argc, argv);
}
