#include "vulture.h"

#include <crypta/graph/rt/brusilov/logger/logger.h>
#include <crypta/graph/rt/lib/sensors/time_shift.h>

namespace NBrusilov {

    TVultureProcessor::TVultureProcessor(TVultureProcessorBase::TConstructionArgs spArgs,
                                         const TVultureConfig& config)
        : TVultureProcessorBase(std::move(spArgs))
        , Config(config)
    {
    }

    TVultureProcessor::TGroupedChunk TVultureProcessor::PrepareGroupedChunk(TString dataSource,
                                                                            TVultureProcessorBase::TManager& stateManager,
                                                                            NBigRT::TMessageBatch data) {
        YT_LOG_DEBUG("Parsed message of type %v", dataSource);
        TVultureProcessor::TGroupedChunk result;
        auto sctx = GetStatsContext({});

        struct Counter {
            size_t alien{0};
            size_t empty{0};
            size_t ids{0};
            size_t protofail{0};
            size_t states{0};
        };
        THashMap<TString, Counter> counterMap{};

        for (auto& packedMessage : data.Messages) {
            packedMessage.Unpack();

            TEventMessage message{};
            TStringBuf skip{};
            for (NFraming::TUnpacker unpacker(packedMessage.Data); unpacker.NextFrame(message, skip);) {
                const auto& source{message.GetSource()};
                {
                    auto mctx{GetStatsContext({{"source", source}})};
                    NCrypta::SetMessageLag(sctx, message);
                    NCrypta::SetMessageLag(mctx, message);
                }
                TVultureEvent event{};

                if (!event.ParseFromString(message.GetBody())) {
                    ++counterMap[source].protofail;
                } else if (const auto& lc{event.GetLocations()};
                    std::find(lc.begin(), lc.end(), Config.GetLocation()) == lc.end()) {
                    ++counterMap[source].alien;
                } else {
                    const auto& associated{event.GetAssociated()};
                    if (const auto& size{associated.GetValueRecords().size()}; size > 0) {
                        ++counterMap[source].states;
                        counterMap[source].ids += size;
                    } else {
                        ++counterMap[source].empty;
                    }

                    const auto vultureKey{TStringBuilder{} << event.GetKeyPrefix() << event.GetId()};
                    auto stateRequest{stateManager.RequestState(vultureKey)};
                    result[std::move(stateRequest)].emplace_back(std::move(associated));
                }
            }
        }
        for (const auto& [source, counter] : counterMap) {
            sctx.Get<NSFStats::TSumMetric<ui64>>("alien").Inc(counter.alien);
            sctx.Get<NSFStats::TSumMetric<ui64>>("empty").Inc(counter.empty);
            sctx.Get<NSFStats::TSumMetric<ui64>>("ids").Inc(counter.ids);
            sctx.Get<NSFStats::TSumMetric<ui64>>("protofail").Inc(counter.protofail);
            sctx.Get<NSFStats::TSumMetric<ui64>>("states").Inc(counter.states);

            auto mctx{GetStatsContext({{"source", source}})};

            mctx.Get<NSFStats::TSumMetric<ui64>>("alien").Inc(counter.alien);
            mctx.Get<NSFStats::TSumMetric<ui64>>("empty").Inc(counter.empty);
            mctx.Get<NSFStats::TSumMetric<ui64>>("ids").Inc(counter.ids);
            mctx.Get<NSFStats::TSumMetric<ui64>>("protofail").Inc(counter.protofail);
            mctx.Get<NSFStats::TSumMetric<ui64>>("states").Inc(counter.states);
        }
        return result;
    }

    void TVultureProcessor::ProcessGroupedChunk(TString dataSource, TVultureProcessor::TGroupedChunk groupedRows) {
        YT_LOG_DEBUG("Process message of type %v size %v", dataSource, groupedRows.size());
        for (auto& [request, rows] : groupedRows) {
            auto& state = request->GetState();
            for (auto& row : rows) {
                state.CopyFrom(row);
            }
        }
    }

    NSFStats::TSolomonContext TVultureProcessor::GetStatsContext(TVector<NSFStats::TSolomonContext::TLabel> labels) {
        labels.push_back({"place", "vulture_processor"});
        return NSFStats::TSolomonContext{SensorsContext.Detached(), labels};
    }
}
