#include <crypta/graph/rt/events/proto/types.pb.h>

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/generic/xrange.h>
#include <util/stream/file.h>
#include <util/string/split.h>

static void GenerateTraits(const TString& name, const THashMap<TStringBuf, TStringBuf>& messages) {
    TFileOutput output(name);

    output << "#pragma once\n\n"
           << "#include <crypta/graph/rt/events/proto/types.pb.h>\n\n";

    // namespace -> classes
    THashMap<TStringBuf, TVector<TStringBuf>> classes;
    for (const auto& p : messages) {
        TStringBuf name = p.first;
        TStringBuf ns;
        if (!name.TryRSplit("::", ns, name)) {
            ns = "NCrypta::NEvent";
        }
        classes[ns].push_back(name);
    }
    // forward declarations
    for (const auto& [ns, names] : classes) {
        output << "namespace " << ns << " {\n";
        for (auto n : names) {
            output << "    class " << n << ";\n";
        }
        output << "}\n\n";
    }
    output << "namespace NCrypta::NEvent::NPrivate {\n"
           << "    template<class T>\n"
           << "    struct TMessageCode;\n\n";

    for (const auto [classname, code] : messages) {
        output << "    template<>\n"
               << "    struct TMessageCode<" << classname << "> {\n"
               << "        static constexpr auto Code = EMessageType::" << code << ";\n"
               << "    };\n\n";
    }
    output << "}\n";
}

void GenerateFactory(const TString& name, const THashMap<TStringBuf, TStringBuf>& messages) {
    TFileOutput output(name);
    // TODO need to extract header name from cpp name
    output << "#include <memory>\n\n"
           << "#include <crypta/graph/rt/events/messages.h>\n"
           << "#include <crypta/graph/rt/events/proto/types.pb.h>\n\n";

    output << "\n"
           << "namespace NCrypta::NEvent::NPrivate {\n"
           << "    THolder<google::protobuf::Message> MakeMessage(const EMessageType code) {\n"
           << "        switch (code) {\n";
    for (const auto [classname, code] : messages) {
        output << "        case EMessageType::" << code << ":\n"
               << "            return MakeHolder<" << classname << ">();\n";
    }
    output << "        case EMessageType::UNDEFINED:\n"
           << "            Y_ASSERT(false);\n"
           << "            break;\n"
           << "        }\n"
           << "        return nullptr;\n"
           << "    }\n"
           << "}\n";
}

// The only argument is output file path
int main(int argc, char** argv) {
    Y_UNUSED(argc);
    Y_ENSURE(argc > 2, "Usage: program <traits> <factory>");
    const TString traits(argv[1]);
    const TString factory(argv[2]);

    THashMap<TStringBuf, TStringBuf> messages;

    for (int i : xrange(NCrypta::NEvent::EMessageType_descriptor()->value_count())) {
        const auto* k = NCrypta::NEvent::EMessageType_descriptor()->value(i);
        TStringBuf schema = k->options().GetExtension(NCrypta::NEvent::Schema);
        if (!schema.empty()) {
            Y_ENSURE(messages.emplace(schema, k->name()).second, "duplicate message class name");
        }
    }

    GenerateTraits(traits, messages);
    GenerateFactory(factory, messages);
    return 0;
}
