#include <array>

#include <crypta/graph/rt/events/generated/factory.h>
#include <util/generic/xrange.h>
#include <util/generic/maybe.h>

#include "events.h"
#include "messages.h"

namespace NCrypta::NEvent {
    namespace {
        using TProcessingOptionsArray = std::array<TMaybe<TProcessingOptions>, GetMessagesCount()>;

        static TProcessingOptionsArray MakeDescriptors() {
            TProcessingOptionsArray result{};
            for (int index : xrange(EMessageType_descriptor()->value_count())) {
                const auto* k{EMessageType_descriptor()->value(index)};
                result[k->number()].ConstructInPlace(TProcessingOptions{
                    .ChunkType = k->options().GetExtension(ChunkType),
                    .KeepInCache = k->options().GetExtension(KeepInCache),
                });
            }
            return result;
        }
    }

    TMessageHolder MakeMessage(const EMessageType code) {
        return NPrivate::MakeMessage(code);
    }

    TMessageHolder MakeMessage(const i64 code) {
        return NPrivate::MakeMessage(static_cast<EMessageType>(code));
    }

    // @brief get processing options by type
    const TProcessingOptions& GetProcessingOptions(const EMessageType type) {
        return GetProcessingOptions(static_cast<i64>(type));
    }

    // @brief get processing options by type
    const TProcessingOptions& GetProcessingOptions(const i64 type) {
        static const auto descriptors{MakeDescriptors()};
        Y_ENSURE_EX(type >= 0 && static_cast<size_t>(type) < descriptors.size(), TUnknownMessageException());
        const auto& desc{descriptors[type]};
        Y_ENSURE_EX(desc.Defined(), TUnknownMessageException());
        return *desc;
    }

    // @brief get processing options by type
    const TMaybe<TProcessingOptions>& TryGetProcessingOptions(const EMessageType type) {
        return TryGetProcessingOptions(static_cast<i64>(type));
    }

    // @brief get processing options by type
    const TMaybe<TProcessingOptions>& TryGetProcessingOptions(const i64 type) {
        static const auto descriptors{MakeDescriptors()};
        if (type >= 0 && static_cast<size_t>(type) < descriptors.size()) {
            return descriptors[type];
        }
        static const TMaybe<TProcessingOptions> empty;
        return empty;
    }
}
