#pragma once

#include <crypta/graph/rt/events/generated/traits.h>
#include <crypta/graph/rt/events/proto/event.pb.h>

#include <util/generic/ptr.h>
#include <util/generic/yexception.h>

namespace NCrypta::NEvent {
    // @brief occurs when trying to unparse unknown message type
    struct TUnknownMessageException : yexception {
    };

    // @brief occurs when message body is malformed and cannot be parsed
    struct TMessageParsingException : yexception {
    };

    using TMessageHolder = THolder<google::protobuf::Message>;

    // @brief return total number of messages
    inline constexpr size_t GetMessagesCount() {
        return EMessageType_MAX + 1;
    }

    // @brief make message instance by type
    TMessageHolder MakeMessage(const EMessageType type);

    // @brief make message instance by type
    TMessageHolder MakeMessage(const i64 type);

    // @brief serialize message to event`s body
    template <class T>
    inline void Pack(TEventMessage& dst, const T& src) {
        Y_PROTOBUF_SUPPRESS_NODISCARD src.SerializeToString(dst.MutableBody());
        dst.SetType(NPrivate::TMessageCode<T>::Code);
    }

    // @brief unpack w/o type checking
    // @note does not check that all required fields are initialized
    // @raises exception if body cannot be parsed
    inline void UnpackAny(const TEventMessage& src, google::protobuf::Message& dst) {
        Y_ENSURE_EX(dst.ParsePartialFromString(src.GetBody()), TMessageParsingException());
    }

    // @brief parse events body to protobuf of appropriate type and return holder to abstract protobuf
    inline TMessageHolder UnpackAny(const TEventMessage& src) {
        auto dst{MakeMessage(src.GetType())};
        Y_ENSURE_EX(dst, TUnknownMessageException());
        UnpackAny(src, *dst);
        return dst;
    }

    // @brief extract message of appropriate type from event
    // @raises exception if type of body match and message cannot be decoded
    // @return true if type match otherwise false
    template <class T>
    inline bool Unpack(const TEventMessage& src, T& dst) {
        if (src.GetType() == NPrivate::TMessageCode<T>::Code) {
            UnpackAny(src, dst);
            return true;
        }
        return false;
    }

    // @brief processing options of message
    struct TProcessingOptions {
        TString ChunkType; // ChunkType
        bool KeepInCache;  // hint for processing that key should be kept in cache
    };

    // @brief get processing options by type
    const TProcessingOptions& GetProcessingOptions(const EMessageType type);

    // @brief get processing options by type
    const TProcessingOptions& GetProcessingOptions(const i64 type);

    // @brief try get processing options by type
    const TMaybe<TProcessingOptions>& TryGetProcessingOptions(const EMessageType type);

    // @brief try get processing options by type
    const TMaybe<TProcessingOptions>& TryGetProcessingOptions(const i64 type);
}
