#include <crypta/graph/rt/fp/herschel/bin/config.pb.h>
#include <crypta/graph/rt/fp/state_processor/state_processor.h>
#include <crypta/graph/rt/services/services.h>
#include <crypta/graph/rt/lib/sensors/build_version.h>

#include <ads/bsyeti/big_rt/lib/consuming_system/consuming_system.h>
#include <ads/bsyeti/big_rt/lib/utility/logging/logging.h>
#include <ads/bsyeti/big_rt/lib/deprecated/sensors/sensors.h>
#include <ads/bsyeti/big_rt/lib/deprecated/services/services.h>
#include <ads/bsyeti/big_rt/lib/supplier/supplier.h>

#include <ads/bsyeti/libs/profiling/solomon/exporter.h>

#include <ads/bsyeti/libs/ytex/client/cache.h>
#include <ads/bsyeti/libs/ytex/client/proto/config.pb.h>
#include <ads/bsyeti/libs/ytex/common/await.h>
#include <ads/bsyeti/libs/ytex/http/server.h>
#include <ads/bsyeti/libs/ytex/http/std_handlers.h>
#include <ads/bsyeti/libs/ytex/program/program.h>

#include <yt/yt/core/misc/shutdown.h>
#include <yt/yt/core/net/address.h>
#include <yt/yt/core/net/config.h>

#include <util/generic/xrange.h>
#include <util/string/join.h>
#include <util/thread/pool.h>


namespace NHerschel {
    using namespace NBigRT;

    void RunHerschelService(const THerschelConfig& config, const NYT::TCancelableContextPtr& cancelableContext) {
        auto solomonExporter = NBSYeti::NProfiling::CreateSolomonExporter(config.GetSolomonExporter());
        solomonExporter->Start();

        auto httpServer = NYTEx::NHttp::CreateServer(config.GetHttpServer());
        NYTEx::NHttp::AddStandardHandlers(
            httpServer, cancelableContext, config,
            BIND([solomonExporter] {
                return NYT::TSharedRef::FromString(NBSYeti::NProfiling::GetSensors(solomonExporter));
            }));
        httpServer->Start();
        INFO_LOG << "Started http server" << Endl;

        auto rootCtx = NBigRT::MakeSolomonContext({});
        NCrypta::BuildVersionSensors(rootCtx);

        NSFStats::TSolomonContext herschelCtx{
            rootCtx, {{"system", "herschel"}}};

        auto ytClients{NYTEx::NRpc::CreateClientsCache(config.GetYtRpc())};
        auto inflightLimiter{CreateProfiledInflightLimiter(herschelCtx, config.GetMaxInflightBytes())};

        auto compositeStateManagerFactory = NYT::New<TCompositeStateManagerFactory>();

        auto herschelStateManagerFactory{
            NYT::New<TSimpleProtoStateManagerFactory<THerschelState>>(
                config.GetStateManagerConfig(),
                TSimpleProtoStateTableOperator({
                    .Table = config.GetStateProcessorConfig().GetTable(),
                    .KeyColumn = "Fp",
                    .ValueColumn = "State",
                    .CodecColumn = "Codec",
                    .Codec = config.GetStateProcessorConfig().GetCodec(),
                }),
                herschelCtx
            )
        };

        TProcessorDescriptors descriptors{
            .HerschelStateDescriptor = compositeStateManagerFactory->Add(
                std::move(herschelStateManagerFactory)
            )
        };

        auto consSystem{
            CreateConsumingSystem({
                .Config = config.GetConsumingSystem(),
                .SuppliersProvider = CreateSupplierFactoriesProvider({.ConfigsRepeated = config.GetSuppliers()}),
                .YtClients = ytClients,
                .ShardsProcessor = [=, &descriptors](TConsumingSystem::IConsumer& consumer) {
                    THerschelProcessor{
                        THerschelProcessor::TConstructionArgs{
                            consumer,
                            config.GetStatefulShardProcessorConfig(),
                            compositeStateManagerFactory,
                            NYTEx::CreateTransactionKeeper(ytClients, TDuration::Zero()),
                            inflightLimiter,
                            herschelCtx,
                            NYT::GetCurrentInvoker()
                        },
                        config.GetStateProcessorConfig(),
                        descriptors
                    }.Run();
                },
                .SensorsContext = herschelCtx,
            })
        };

        consSystem->Run();
        NYTEx::WaitFor(cancelableContext);
        INFO_LOG << "Stopping consuming system\n";
        NYT::NConcurrency::WaitUntilSet(consSystem->Stop());
    }


    class THerschelProgram: public NYTEx::TProgram<THerschelConfig> {
    public:
        THerschelProgram()
            : NYTEx::TProgram<THerschelConfig>("herschel") {
        }

        int DoRun(const NYT::TCancelableContextPtr& context) override {
            RunHerschelService(Config(), context);
            return 0;
        }
    };
}

int main(int argc, const char** argv) {
    return NYTEx::RunProgram<NHerschel::THerschelProgram>(argc, argv);
}
