#include "features_calculator.h"

namespace NCrypta {
    THolder<uatraits::detector> GetUATraitsDetector() {
        THolder<uatraits::detector> UATraitsDetector;

        auto browser = NResource::Find("browser.xml");
        auto profiles = NResource::Find("profiles.xml");
        auto extra = NResource::Find("extra.xml");
        UATraitsDetector.Reset(new uatraits::detector(
            browser.data(), browser.size(),
            profiles.data(), profiles.size(),
            extra.data(), extra.size()));

        return UATraitsDetector;
    };

    const THerschelFeaturesCalculator::TKeyTypeFeaturesByKeyType THerschelFeaturesCalculator::KeyTypeFeaturesByKeyType = {
        .Ip = {
            .UseBrowser = 0,
            .UseDevice = 0,
            .UseIp = 1,
            .UseUseragent = 0,
        },
        .IpUseragent = {
            .UseBrowser = 0,
            .UseDevice = 0,
            .UseIp = 1,
            .UseUseragent = 1,
        },
    };

    THerschelFeaturesCalculator::THerschelFeaturesCalculator(const NGeobase::TLookup& lookup)
        : Lookup(lookup)
        , UATraitsDetector(GetUATraitsDetector()) {
    };

    TFeatures THerschelFeaturesCalculator::GetFeatures(const TStringBuf& ip, const TStringBuf& useragent, const THerschelStats& herschelStats) const {
        const auto ipString = std::string(ip.begin(), ip.end());
        uatraits::detector::result_type useragentParsed;
        UATraitsDetector->detect(std::string(useragent.begin(), useragent.end()), useragentParsed);

        auto features = TFeatures{
            .FloatFeatures = {
                0, // use_browser
                0, // use_device
                0, // use_ip
                0, // use_useragent
                static_cast<float>(herschelStats.GetGaidCount()), // gaidhash_cnt
                static_cast<float>(herschelStats.GetIdfaCount()), // idfahash_cnt
                static_cast<float>(herschelStats.GetYandexuidCount()), // yandexuid_cnt
                static_cast<float>(herschelStats.GetGaidCount() + herschelStats.GetIdfaCount() + herschelStats.GetYandexuidCount()), // gaid_idfa_yandexuid_sum
                static_cast<float>(Lookup.GetBasicTraitsByIp(ipString).IsMobile()), // is_mobile_ip
                static_cast<float>(NIPREG::TAddress::ParseAny(ipString).IsIPv4()), // is_ip_v4
            },
            .CategoricalFeatures = {
                TString(std::to_string(Lookup.GetRegionByIp(ipString).GetId())), // region
                TString(useragentParsed["BrowserName"]), // browser_name
                TString(useragentParsed["BrowserVersion"]), // browser_version
                TString(useragentParsed["BrowserBase"]), // browser_base
                TString(useragentParsed["OSFamily"]), // os_family
                TString(useragentParsed["OSVersion"]), // os_version
                TString(useragentParsed["DeviceName"]), // device_name
            },
        };

        return features;
    };

    void THerschelFeaturesCalculator::UpdateKeyTypeFeatures(TFeatures& features, const THerschelFeaturesCalculator::TKeyTypeFeatures& keyFeatures) const {
        features.FloatFeatures[0] = keyFeatures.UseBrowser;
        features.FloatFeatures[1] = keyFeatures.UseDevice;
        features.FloatFeatures[2] = keyFeatures.UseIp;
        features.FloatFeatures[3] = keyFeatures.UseUseragent;
    };
}
