#include <crypta/graph/rt/events/events.h>
#include <crypta/graph/rt/events/proto/event.pb.h>
#include <crypta/graph/rt/events/proto/fp.pb.h>
#include <crypta/graph/rt/fp/rows_processor/proto/config.pb.h>

#include <crypta/graph/soup/config/cpp/soup_config.h>
#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/lib/native/identifiers/lib/id_types/all.h>

#include <google/protobuf/util/message_differencer.h>

#include <library/cpp/framing/unpacker.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/testing/common/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/vector.h>
#include <util/stream/file.h>

#include "rows_processor.h"
#include "algo.h"

using namespace NResharder;
using namespace NIdentifiers;
using google::protobuf::util::MessageDifferencer;

namespace NCrypta {
Y_UNIT_TEST_SUITE(RowsProcessorTestSuite) {
    Y_UNIT_TEST(TestParseUnknown) {
        const TString configJson{
            R"({
                "MessageType": "SOME_UNKNOWN_MESSAGE",
                "Parser": {
                    "FromEmpty": {
                    }
                }
            })"};

        UNIT_ASSERT_EXCEPTION(MakeRowsProcessor(
            NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson), 3, "someQueue"), yexception);
    }

    Y_UNIT_TEST(TestProcessTskv) {
        const TString configJson{
            R"({
                "ChoiceType": "1",
                "MessageType": "FP",
                "Parser": {
                    "FromBsRtbLog": {
                        "Lifetime": 0,
                        "GeoBasePath": ")" + BinaryPath("geobase/data/v6/geodata6.bin") + R"("
                    }
                },
                "ReshardingModule": 3,
                "ShardingAlgorithm": "Fingerprint"
            })"};

        TFileInput fileStream{SRC_("bs-event-log.tskv")};

        auto rowsProcessor{MakeRowsProcessor(NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson), 3, "someQueue")};

        // metrics
        NSFStats::TStats stats{};
        NSFStats::TSolomonContext ctx{stats, {}};

        NBigRT::TRowsBatch rows;
        NBigRT::TRowMeta meta;
        THashMap<TString, TVector<NBigRT::TAggregatedRow>> aggregatedRows;
        rowsProcessor->Parse(ctx, meta, fileStream.ReadAll(), rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 228);
        rowsProcessor->Process(ctx, TInstant::Now(), rows);
        rowsProcessor->Aggregate(ctx, rows, aggregatedRows);
        int rowsCount = 0;
        THashMap<NCrypta::NEvent::EHerschelVersion, size_t> counter;
        for (const auto& [destinationQueue, rows] : aggregatedRows) {
            for (const auto& row : rows) {
                UNIT_ASSERT(row.Shard < 3);
                UNIT_ASSERT(TInstant::Zero() < row.TimeStamp);
                ++rowsCount;

                const auto& msg{dynamic_cast<const NCrypta::NEvent::TFpEvent*>(row.Message.Get())};
                ++counter[msg->GetFingerprint().GetVersion()];
            }
        }
        UNIT_ASSERT_VALUES_EQUAL(rowsCount, 228);
        UNIT_ASSERT_VALUES_EQUAL(counter[NCrypta::NEvent::EHerschelVersion::NAIVE], 114);
        UNIT_ASSERT_VALUES_EQUAL(counter[NCrypta::NEvent::EHerschelVersion::MODEL_IP_UA_V0], 114);
    }

    Y_UNIT_TEST(TestShardingAlgorithm) {
        const auto& shardingAlgorithm{MakeSharder("Fingerprint")};
        NBigRT::TRow row{.Message = NCrypta::NEvent::MakeMessage(NCrypta::NEvent::EMessageType::FP)};
        const auto& msg{dynamic_cast<NCrypta::NEvent::TFpEvent*>(row.Message.Get())};
        const auto& fp{msg->MutableFingerprint()};
        {
            // Change fp stamp check is shard changed;
            fp->SetUserIP("01234567890");
            fp->SetUAHash(1234);
            fp->SetVersion(NCrypta::NEvent::EHerschelVersion::NAIVE);
            UNIT_ASSERT_VALUES_EQUAL(shardingAlgorithm->Shard(row), 260404154u);
        }
        {
            // Change fp stamp check is shard changed;
            fp->SetVersion(NCrypta::NEvent::EHerschelVersion::MODEL_IP_UA_V0);
            UNIT_ASSERT_VALUES_EQUAL(shardingAlgorithm->Shard(row), 3841999302u);
        }
        {
            // Change fp stamp check is shard changed;
            fp->SetUserIP("01234567891");
            UNIT_ASSERT_VALUES_EQUAL(shardingAlgorithm->Shard(row), 1843787499u);
        }
        {
            // Change fp stamp check is shard changed;
            fp->SetUserIP("72123124083");
            fp->SetUAHash(123456);
            UNIT_ASSERT_VALUES_EQUAL(shardingAlgorithm->Shard(row), 132486445u);
        }
        {
            // add some ids check is shard NOT changed;
            const auto id{NIdentifiers::TYandexuid("1234567891532376011")};
            *(msg->AddIds()) = id.ToProto();

            UNIT_ASSERT_VALUES_EQUAL(shardingAlgorithm->Shard(row), 132486445u);
        }
        {
            // Change fp stamp check is shard changed;
            fp->SetUserIP("1234567891532376011");
            fp->SetUAHash(3456789);
            UNIT_ASSERT_VALUES_EQUAL(shardingAlgorithm->Shard(row), 1028397698u);
        }
    }
}}
