#pragma once

#include <util/datetime/base.h>
#include <util/generic/ptr.h>
#include <util/generic/strbuf.h>
#include <util/generic/vector.h>

#include <crypta/graph/rt/events/events.h>
#include <crypta/graph/rt/events/proto/event.pb.h>
#include <crypta/graph/rt/events/proto/fp.pb.h>
#include <crypta/graph/rt/events/proto/vulture.pb.h>

#include <crypta/graph/rt/fp/model/catboost_applier.h>
#include <crypta/graph/rt/fp/model/features_calculator.h>

#include <crypta/graph/rt/fp/state_processor/proto/state.pb.h>
#include <crypta/graph/rt/lib/debounce_cache/debounce_cache.h>
#include <crypta/graph/rt/lib/qyt_batch_multipacker/qyt_batch_multipacker.h>

#include <library/cpp/safe_stats/safe_stats.h>
#include <library/cpp/framing/unpacker.h>
#include <library/cpp/geobase/lookup.hpp>

#include <ads/bsyeti/big_rt/lib/processing/shard_processor/stateful/processor.h>
#include <ads/bsyeti/big_rt/lib/processing/state_manager/simple_proto/factory.h>
#include <ads/bsyeti/big_rt/lib/processing/state_manager/simple_proto/manager.h>
#include <ads/bsyeti/big_rt/lib/processing/state_manager/composite/factory.h>
#include <ads/bsyeti/big_rt/lib/processing/state_manager/composite/manager.h>
#include <ads/bsyeti/protos/vulture_messages.pb.h>

namespace {
    static const TString geobasePath{"/herschel/geodata6.bin"};
    static const TString modelPath{"/herschel/mlh.bin"};
    static const TString thresholdsPath{"/herschel/thresholds.json"};
}

namespace NHerschel {
    using namespace NBigRT;

    using TEventMessage = NCrypta::NEvent::TSimpleEventMessage;
    using TVultureEvent = NCrypta::NEvent::TVultureEvent;
    using TFpEvent = NCrypta::NEvent::TFpEvent;
    using TTimestamp = ui64;
    using TAssociatedUids = NVulture::TAssociatedUids;

    struct THerschelGroupedChunk {
        TFpEvent event;
        TTimestamp TimeStamp; 
    };

    using THerschelStateDescriptor = TStateDescriptor<TSimpleProtoStateManagerFactory<THerschelState>>;

    struct TProcessorDescriptors {
        THerschelStateDescriptor HerschelStateDescriptor;
    };

    using THerschelProcessorBase = TCompositeStatefulShardProcessor<THerschelGroupedChunk>;

    class THerschelProcessor : public THerschelProcessorBase {
    private:
        struct TBanCounterMetric {
            ui64 unBannedFps{0};
            ui64 bannedFps{0};
            ui64 changed{0};
            ui64 updated{0};
            ui64 skipped{0};
            ui64 unBanTime{0};
            ui64 hysteresisTime{0};
        };

    public:
        THerschelProcessor(THerschelProcessorBase::TConstructionArgs spArgs,
                           const TStateProcessorConfig& config,
                           TProcessorDescriptors descriptors)
            : THerschelProcessorBase(std::move(spArgs))
            , Config(config)
            , Descriptors(std::move(descriptors))
            , GeoBase(geobasePath)
            , BruPacker(Config.GetBrusilov())
            , Debounce("deb", config.GetDebounceCfg())
            , Banned("ban", config.GetBanCfg().GetDebounceCfg())
            , Model(GeoBase, modelPath, thresholdsPath) {
        }

        TGroupedChunk PrepareGroupedChunk(TString dataSource, THerschelProcessorBase::TManager& stateManager, TMessageBatch data) override;
        void ProcessGroupedChunk(TString dataSource, TGroupedChunk groupedRows) override;
        NYT::TFuture<TPrepareForAsyncWriteResult> PrepareForAsyncWrite() override;

    private:
        const TStateProcessorConfig& Config;
        const TProcessorDescriptors Descriptors;
        const NGeobase::TLookup GeoBase;
        NRtSklejka::TQytBatchMultiPacker BruPacker;
        NRtCrypta::TDebounceCache Debounce;
        NRtCrypta::TDebounceCache Banned;
        NCrypta::THerschelCatboostApplier Model;

        bool BanPopularFps(NCrypta::NEvent::TFingerprint key, THerschelState& state, const ui64 newIdsCounter, TBanCounterMetric* counter);
        bool BanPopularFpsNaive(THerschelState& state, const ui64 newIdsCounter, TBanCounterMetric* counter);
        bool BanPopularFpsMlIpUaV0(THerschelState& state, [[maybe_unused]] const ui64 newIdsCounter, TBanCounterMetric* counter);
        ui64 UpdateHershelStateIds(THerschelState& state, size_t limit, const TChunk& rows);
        void ClearState(THerschelState& state);
        void ClearVultureState(const TString& key);
        void UpdateVultureState(const TString& key, THerschelState& herschel);
        TEventMessage CreateVultEvent(const TString& key, const TAssociatedUids& associated);
    };
}
