#pragma once

#include "manager.h"

#include <library/cpp/iterator/zip.h>


namespace NRtCrypta {
    template <typename TState>
    TVultureStateManager<TState>::TVultureStateManager(
        TStateManagerConfig stateConfig,
        TDebounceConfig debConfig,
        TVultureStateTableOperator stateOperator,
        NSFStats::TSolomonContext sensorsContext
    )
        : TGenericStateManager<TString, TState>(std::move(stateConfig), sensorsContext)
        , StateOperator(std::move(stateOperator))
        , Debounce(DebounceId, debConfig)
        , SensorsContext(sensorsContext)
    {
    }

    template <typename TState>
    TVultureStateManager<TState>::TVultureStateManager(
        TStateManagerConfig stateConfig,
        TDebounceConfig debConfig,
        TVultureStateTableOperator stateOperator,
        const NYT::NProfiling::TTagSet& tags
    )
        : TVultureStateManager(std::move(stateConfig), std::move(debConfig),
                               std::move(stateOperator), MakeSolomonContext(tags))
    {
    }

    template <typename TState>
    NYT::TFuture<
        TVector<TBaseStateRequestPtr>
    > TVultureStateManager<TState>::LoadStates(
        TVector<TBaseRequestPtr> requests,
        NYT::NApi::IClientPtr client,
        NYT::NTransactionClient::TTimestamp timestamp
    ) {
        Y_UNUSED(timestamp);

        auto ids = TVector<TStateId>(
            Reserve(requests.size())
        );

        for (const auto& request: requests) {
            ids.push_back(request->GetStateId());
        }

        return StateOperator.Load(
            std::move(client), std::move(ids)
        ).Apply(BIND([
            stateOperator = StateOperator,
            requests = std::move(requests)
        ](const TVector<TString>& rows) {
            auto baseRequests = TVector<TBaseStateRequestPtr>(
                Reserve(requests.size())
            );

            auto states = stateOperator.template Parse<TState>(rows);

            for (auto&& [request, state]: Zip(requests, states)) {
                request->GetState() = std::move(state);
                baseRequests.push_back(std::move(request));
            }

            return baseRequests;
        }));
    }

    template <typename TState>
    TBaseStateManager::TStateWriter
    TVultureStateManager<TState>::WriteStates(
        TVector<TBaseRequestPtr> requests
    ) {
        auto states = TVector<TState>(
            Reserve(requests.size())
        );

        auto ids = TVector<TStateId>(
            Reserve(requests.size())
        );

        size_t writeCounter{0};
        size_t removeCounter{0};

        for (auto&& request: requests) {
            const auto& stateId{request->GetStateId()};
            const size_t stateSize{static_cast<size_t>(request->GetState().ByteSize())};
            const TString debId{(stateSize ? "u" : "r") + stateId};

            if (0 == stateSize) {
                ++removeCounter;
            } else {
                ++writeCounter;
            }

            if (Debounce.Pull(debId)) {
                continue;
            }
            ids.push_back(stateId);
            states.push_back(request->GetState());
            Debounce.Push(debId);
        }

        {
            NSFStats::TSolomonContext sc(SensorsContext.Detached(), {{"place", "vulture_manager"}});
            sc.Get<NSFStats::TSumMetric<ui64>>("write").Inc(writeCounter);
            sc.Get<NSFStats::TSumMetric<ui64>>("remove").Inc(removeCounter);
            sc.Get<NSFStats::TSumMetric<ui64>>("final").Inc(ids.size());
            sc.Get<NSFStats::TSumMetric<ui64>>("skipped").Inc(writeCounter + removeCounter - ids.size());
        }

        auto serialized = StateOperator.Serialize(std::move(states));
        return [
            ids = std::move(ids),
            serialized = std::move(serialized),
            stateOperator = StateOperator
        ](NYT::NApi::ITransactionPtr tx) {
            stateOperator.Write(std::move(tx), ids, serialized);
        };
    }

    template <typename TState>
    size_t TVultureStateManager<TState>::SpaceUsed(const TState& state) {
        return state.SpaceUsedLong();
    }
}
