#include "operator.h"

namespace NRtCrypta {
    TVultureStateTableOperator::TVultureStateTableOperator(
        TOptions options
    )
        : Options(std::move(options))
    {
    }

    NYT::TFuture<TVector<TString>> TVultureStateTableOperator::Load(
        NYT::NApi::IClientPtr client, const TVector<TString>& stringKeys
    ) const {
        if (Options.WriteOnlyState) {
            // no read from state table
            TVector<TString> emptyResponse(stringKeys.size());
            return NYT::MakeFuture<TVector<TString>>(std::move(emptyResponse));
        }

        auto nameTable = NYT::New<NYT::NTableClient::TNameTable>();
        i32 keyField = nameTable->GetIdOrRegisterName(Options.KeyColumn);

        auto rowBuffer = NYT::New<NYT::NTableClient::TRowBuffer>();
        auto keys = TVector<NYT::NTableClient::TUnversionedRow>(
            Reserve(stringKeys.size())
        );

        for (auto& key: stringKeys) {
            NYT::NTableClient::TUnversionedRowBuilder builder;
            builder.AddValue(
                NYT::NTableClient::MakeUnversionedStringValue(TStringBuilder{} << Options.KeyPrefix << key, keyField)
            );

            keys.push_back(rowBuffer->CaptureRow(builder.GetRow()));
        }

        const auto range = NYT::MakeSharedRange(
            std::move(keys), std::move(rowBuffer)
        );

        NYT::NApi::TLookupRowsOptions options;
        options.Timeout = TDuration::Seconds(10);
        options.KeepMissingRows = true;

        return client->LookupRows(
            Options.Table, nameTable, range, options
        ).Apply(BIND([
            count = stringKeys.size(),
            valueColumn = Options.ValueColumn,
            Codec = Options.Codec
        ](NYT::NApi::IUnversionedRowsetPtr rowset) {
            Y_ENSURE(rowset->GetRows().size() == count);

            auto schema = rowset->GetSchema();
            i32 valueIndex = schema->GetColumnIndexOrThrow(valueColumn);

            auto isStringLikeType = [](const auto& data, bool allowOptional) {
                switch (data.Type) {
                case NYT::NTableClient::EValueType::String:
                    return true;

                case NYT::NTableClient::EValueType::Null:
                    return allowOptional;

                default:
                    return false;
                }
            };

            TBuffer buffer;

            auto values = TVector<TString>(Reserve(count));

            for (const auto& row: rowset->GetRows()) {
                values.emplace_back();

                if (!row) {
                    continue;
                }

                const auto& valueData = row[valueIndex];
                Y_ENSURE(isStringLikeType(valueData, false));

                if (!valueData.Length) {
                    continue;
                }

                auto valueDataView = TStringBuf(
                    valueData.Data.String, valueData.Length
                );

                const auto codecPtr = NBlockCodecs::Codec(Codec);

                buffer.Clear();
                codecPtr->Decode(valueDataView, buffer);

                valueDataView = TStringBuf(buffer.data(), buffer.size());

                values.back() = TString(valueDataView);
            }

            return values;
        }));
    }


    void TVultureStateTableOperator::Write(
        NYT::NApi::ITransactionPtr tx,
        const TVector<TString>& keys, const TVector<TString>& values
    ) const {
        auto nameTable = NYT::New<NYT::NTableClient::TNameTable>();
        i32 keyField = nameTable->GetIdOrRegisterName(Options.KeyColumn);
        i32 valueField = nameTable->GetIdOrRegisterName(Options.ValueColumn);
        i32 eraField = nameTable->GetIdOrRegisterName(Options.EraColumn);
        i32 rtField = nameTable->GetIdOrRegisterName(Options.RtColumn);

        auto rowBuffer = NYT::New<NYT::NTableClient::TRowBuffer>();
        auto rows = TVector<NYT::NApi::TRowModification>();

        auto makeUnversionedValue = [&rowBuffer](i32 field, TStringBuf value) {
            return rowBuffer->CaptureValue(
                NYT::NTableClient::MakeUnversionedStringValue(value, field)
            );
        };

        auto makeUnversionedUi64Value = [&rowBuffer](i32 field, ui64 value) {
            return rowBuffer->CaptureValue(
                NYT::NTableClient::MakeUnversionedUint64Value(value, field));
        };

        TBuffer buffer;

        const auto& ChangeRow{
            [&](const TString& key, const TString& value) {
                auto row = rowBuffer->AllocateUnversioned(4);  // key, value, era, rt
                row[keyField] = makeUnversionedValue(keyField, TStringBuilder{} << Options.KeyPrefix << key);
                auto valueView = TStringBuf(value);

                {
                    buffer.Clear();

                    const auto codecPtr = NBlockCodecs::Codec(Options.Codec);
                    codecPtr->Encode(valueView, buffer);

                    valueView = TStringBuf(buffer.data(), buffer.size());
                }

                row[valueField] = makeUnversionedValue(valueField, valueView);

                {
                    const ui64 now{TInstant::Now().Seconds()};
                    row[eraField] = makeUnversionedUi64Value(eraField, now);
                    row[rtField] = makeUnversionedUi64Value(rtField, now);
                }

                rows.push_back({
                    NYT::NApi::ERowModificationType::Write,
                    row.ToTypeErasedRow(), NYT::NTableClient::TLockMask()
                });
            }
        };

        const auto& RemoveRow{
            [&](const TString& key, [[maybe_unused]] const TString& value) {
                auto row = rowBuffer->AllocateUnversioned(1);  // key
                row[keyField] = makeUnversionedValue(keyField, TStringBuilder{} << Options.KeyPrefix << key);
                rows.push_back({
                    NYT::NApi::ERowModificationType::Delete,
                    row.ToTypeErasedRow(), NYT::NTableClient::TLockMask()
                });
            }
        };

        for (auto [key, value] : Zip(keys, values)) {
            if (value.empty()) {
                RemoveRow(key, value);
            } else {
                ChangeRow(key, value);
            }
        }

        const auto range = NYT::MakeSharedRange(
            std::move(rows), std::move(rowBuffer)
        );

        tx->ModifyRows(Options.Table, nameTable, range);
    }
}
