#include "wrapper.h"

#include <yt/yt/client/table_client/helpers.h>

using namespace NYTRpc;

namespace NCrypta {
    TYtCryptaIdProtosSaver::TYtCryptaIdProtosSaver(const TString& table)
        : Table(table)
        , Context() {
    }

    TYtCryptaIdProtosSaver::TStaticContext::TStaticContext()
            : ModifyRowsOptions()
            , NameTable(NYT::New<NYT::NTableClient::TNameTable>())
            , KeyField(NameTable->GetIdOrRegisterName("Id"))
            , ValueField(NameTable->GetIdOrRegisterName("CryptaId")) {
        ModifyRowsOptions.RequireSyncReplica = true;
    }

    void TYtCryptaIdProtosSaver::SaveRowset(const TCryptaIdPairVector& rows, NYT::NApi::IClientPtr client) {

        auto tx_future = client->StartTransaction(NYT::NTransactionClient::ETransactionType::Tablet);
        auto tx = tx_future.Get().ValueOrThrow();

        auto rowBuffer{NYT::New<NYT::NTableClient::TRowBuffer>()};

        TVector<NYT::NApi::TRowModification> ytRows;

        for (const auto& item : rows) {
            auto ytRow = rowBuffer->AllocateUnversioned(2);
            ytRow[0] = rowBuffer->CaptureValue(NYT::NTableClient::MakeUnversionedStringValue(
                        item.first.Serialize(),
                        Context.KeyField
            ));

            TString serialized;
            Y_PROTOBUF_SUPPRESS_NODISCARD item.second.ToProto().SerializeToString(&serialized);
            ytRow[1] = rowBuffer->CaptureValue(NYT::NTableClient::MakeUnversionedStringValue(
                        serialized,
                        Context.ValueField
            ));
            ytRows.push_back({NYT::NApi::ERowModificationType::Write, ytRow.ToTypeErasedRow(), NYT::NTableClient::TLockMask()});
        }
        tx->ModifyRows(
                Table,
                Context.NameTable,
                MakeSharedRange(std::move(ytRows), std::move(rowBuffer)),
                Context.ModifyRowsOptions
        );

        tx->Commit().Get().ValueOrThrow();
    }

    TYtCryptaIdProtosLoader::TStaticContext::TStaticContext(TDuration timeout, bool syncRead)
            : LookupRowsOptions()
            , NameTable(NYT::New<NYT::NTableClient::TNameTable>())
            , KeyField(NameTable->GetIdOrRegisterName("Id"))
            , ValueField(NameTable->GetIdOrRegisterName("CryptaId"))
            , Codec("none") {
        LookupRowsOptions.KeepMissingRows = true;
        LookupRowsOptions.Timestamp = syncRead ? NYT::NTransactionClient::SyncLastCommittedTimestamp
                                               : NYT::NTransactionClient::AsyncLastCommittedTimestamp;
        LookupRowsOptions.Timeout = timeout;
        LookupRowsOptions.ColumnFilter = NYT::NTableClient::TColumnFilter({KeyField, ValueField});
    }

    TYtCryptaIdProtosLoader::TYtCryptaIdProtosLoader(const TString& table, const TDuration timeout, const bool syncRead)
        : Table(table)
        , Context(timeout, syncRead) {
    }

    TVector<std::pair<TBuffer,TBuffer>> TYtCryptaIdProtosLoader::ParseRows(NYT::NApi::IUnversionedRowsetPtr rowset) {
        const auto schema{rowset->GetSchema()};
        const int valueFieldId{schema->GetColumnIndexOrThrow(TStringBuf("CryptaId"))};
        const int keyFieldId{schema->GetColumnIndexOrThrow(TStringBuf("Id"))};
        TVector<std::pair<TBuffer, TBuffer>> rowsData(rowset->GetRows().size());
        size_t index{0};
        // TODO does need mark empty protobuf response?
        for (auto& row : rowset->GetRows()) {
            if (row) {
                const TStringBuf valueInfo{NYT::NTableClient::FromUnversionedValue<TStringBuf>(row[valueFieldId])};
                rowsData[index].first.Assign(valueInfo.data(), valueInfo.size());
                rowsData[index].first.ShrinkToFit();

                const TStringBuf keyInfo{NYT::NTableClient::FromUnversionedValue<TStringBuf>(row[keyFieldId])};
                rowsData[index].second.Assign(keyInfo.data(), keyInfo.size());
                rowsData[index].second.ShrinkToFit();
            }
            ++index;
        }
        return rowsData;
    }

    TCryptaIdProtoPackVector TYtCryptaIdProtosLoader::BuildCryptaIdProtos(const TVector<std::pair<TBuffer,TBuffer>>& rows) {
        TCryptaIdProtoPackVector result{rows.size()};
        size_t index{0};
        for (auto& row : rows) {
            if (!row.first.Empty()) {
                Y_PROTOBUF_SUPPRESS_NODISCARD result[index].MutableCryptaId()->ParsePartialFromArray(row.first.data(), row.first.size());
            }
            if (!row.second.Empty()) {
                Y_PROTOBUF_SUPPRESS_NODISCARD result[index].MutableId()->ParsePartialFromArray(row.second.data(), row.second.size());
            }
            ++index;
        }

        return result;
    }

    TCryptaIdProtoPackVector TYtCryptaIdProtosLoader::BuildCryptaIdProtos(NYT::NApi::IUnversionedRowsetPtr rowset) {
        return BuildCryptaIdProtos(ParseRows(rowset));
    }

    NYT::TFuture<NYT::NApi::IUnversionedRowsetPtr> TYtCryptaIdProtosLoader::LoadRowsetFuture(
            const TVector<NIdentifiers::TGenericID>& requestedKeys, NYT::NApi::IClientPtr client) {
        using NYT::NTableClient::MakeUnversionedStringValue;
        using NYT::NTableClient::TRowBuffer;
        using NYT::NTableClient::TUnversionedRow;
        using NYT::NTableClient::TUnversionedRowBuilder;
        try {
            auto rowBuffer{NYT::New<TRowBuffer>()};

            TVector<TUnversionedRow> keys;
            keys.reserve(requestedKeys.size());

            for (const auto& id : requestedKeys) {
                TUnversionedRowBuilder builder;
                const TString serialized{id.Serialize()};
                TStringBuf buff{serialized.data(), serialized.size()};
                builder.AddValue(NYT::NTableClient::MakeUnversionedStringValue(buff, Context.KeyField));
                keys.emplace_back(rowBuffer->CaptureRow(builder.GetRow()));
            }
            const auto range{NYT::MakeSharedRange(std::move(keys), std::move(rowBuffer))};

            NYT::TFuture<IUnversionedRowsetPtr> ytFuture{
                client->LookupRows(Table, Context.NameTable, range, Context.LookupRowsOptions)};
            return ytFuture;
        } catch (...) {
            ERROR_LOG << CurrentExceptionMessage() << "\n";
            return {};
        }
    }

    TCryptaIdProtoPackVector TYtCryptaIdProtosLoader::LoadRowset(
            const TVector<NIdentifiers::TGenericID>& requestedKeys, NYT::NApi::IClientPtr client) {
        auto responseFuture{LoadRowsetFuture(requestedKeys, client)};
        auto unparsedResponse{responseFuture.Get().ValueOrThrow()};
        return BuildCryptaIdProtos(unparsedResponse);
    }
}
