#include <library/cpp/testing/unittest/registar.h>

#include <crypta/graph/rt/sklejka/cid_resolver/lib/resolver.h>
#include <crypta/lib/native/identifiers/lib/id_types/all.h>

#include <library/cpp/iterator/zip.h>
#include <util/generic/string.h>

using namespace NCrypta;

Y_UNIT_TEST_SUITE(IdentificationTestSuite) {
    template <class T>
    using TChunkResponse = ICryptaIdResolver::TChunkResponse<T>;
    using TCryptaId = ICryptaIdResolver::TCryptaId;
    using TGenericID = NIdentifiers::TGenericID;

    Y_UNIT_TEST(Simple) {
        TVector<TGenericID> ids{};
        {
            ids.emplace_back("duid", "1109604822465415182");
            ids.emplace_back("duid", "11150350001712494599");
            ids.emplace_back("duid", "14353619782832511914");
            ids.emplace_back("gaid", "b3525951-c5f4-816d-f75c-dde301d5fe92");
            ids.emplace_back("gaid", "cabceb05-32e5-97d7-259c-ca5e339ccfc4");
            ids.emplace_back("gaid", "ce400dfc-5d06-17a7-e56f-9500188a1148");
            ids.emplace_back("idfa", "0214DE45-6E94-8A91-3158-4685F17FD349");
            ids.emplace_back("idfa", "509110A8-0A39-261D-6036-632B3A647A9D");
            ids.emplace_back("idfa", "FB225E43-54BF-2AB2-9E21-DD75BF950EE1");
            ids.emplace_back("yandexuid", "376779711439526274");
            ids.emplace_back("yandexuid", "639223371098951143");
            ids.emplace_back("yandexuid", "770718541501441773");
            // duplicate
            ids.emplace_back("yandexuid", "770718541501441776");
            ids.emplace_back("yandexuid", "770718541501441776");
            ids.emplace_back("yandexuid", "770718541501441776");
            ids.emplace_back("gaid", "ce400dfc-5d06-17a7-e56f-9500188a1148");
            ids.emplace_back("gaid", "ce400dfc-5d06-17a7-e56f-9500188a1148");
            ids.emplace_back("gaid", "ce400dfc-5d06-17a7-e56f-9500188a1148");
            ids.emplace_back("idfa", "0214DE45-6E94-8A91-3158-4685F17FD349");
            ids.emplace_back("idfa", "0214DE45-6E94-8A91-3158-4685F17FD349");
            ids.emplace_back("idfa", "0214DE45-6E94-8A91-3158-4685F17FD349");
            ids.emplace_back("idfa", "0214DE45-6E94-8A91-3158-4685F17FD349");
        };

        ui32 uniqsSeen{0};
        const auto rawRequester{
            [&uniqsSeen](const TVector<TGenericID>& uniqs, const ui32 attempt) {
                if (attempt < 2) {
                    throw yexception{} << "Fake No response";
                }

                // check deduplication
                uniqsSeen += uniqs.size();

                TChunkResponse<TCryptaId> response(uniqs.size(), Nothing());
                for (auto [srcId, dst] : Zip(uniqs, response)) {
                    auto src{srcId.GetValue()};
                    if (const size_t n{srcId.Hash()}; n & 1) {
                        dst = n & 0xFF;
                    } else {
                        // let it be not found
                        // Cerr << srcId.Hash() << " " << srcId.GetValue() << "\n";
                        continue;
                    }
                }
                return response;
            }};

        auto cache = NYT::New<TIdentificationCache<TCryptaId>>("test", 1e6, TDuration::Hours(1));
        NSFStats::TStats stats;
        NSFStats::TSolomonContext ctx{stats};

        ui32 batchSize{3};
        {
            const auto lambda{
                [&]() {
                    return DoIdentificationRequest<TCryptaId>(ids, rawRequester, 2, batchSize, *cache, ctx);
                }};
            UNIT_ASSERT_EXCEPTION(lambda(), ICryptaIdResolver::TNoResponseException);
        }
        {
            const auto lambda{
                [&]() {
                    return DoIdentificationRequest<TCryptaId>(ids, rawRequester, 3, batchSize, *cache, ctx);
                }};
            const TChunkResponse<TCryptaId> etalon{
                // duids
                Nothing(), 9, 237,
                // gaids
                77, 251, 13,
                // idfas
                19, 241, Nothing(),
                // yandexuids
                119, 49, Nothing(),
                // duplicate
                Nothing(), Nothing(), Nothing(),
                13, 13, 13,
                19, 19, 19, 19};
            UNIT_ASSERT_VALUES_EQUAL(lambda(), etalon);
            UNIT_ASSERT_VALUES_EQUAL(uniqsSeen, 13); // check deduplication
        }
    }
}
