#include <crypta/graph/rt/sklejka/michurin/bin/config.pb.h>

#include <crypta/graph/rt/sklejka/michurin/proto/state.pb.h>

#include <crypta/graph/rt/sklejka/michurin/processors/cryptaid_processor.h>
#include <crypta/graph/rt/sklejka/michurin/processors/michurin_processor.h>
#include <crypta/graph/rt/sklejka/michurin/processors/michurin_state_manager.h>
#include <crypta/graph/rt/sklejka/michurin/processors/michurin_state_manager_factory.h>
#include <crypta/graph/rt/sklejka/michurin/processors/michurin_state_operator.h>

#include <ads/bsyeti/big_rt/lib/consuming_system/consuming_system.h>
#include <ads/bsyeti/big_rt/lib/utility/logging/logging.h>
#include <ads/bsyeti/big_rt/lib/deprecated/sensors/sensors.h>
#include <ads/bsyeti/big_rt/lib/deprecated/services/services.h>
#include <ads/bsyeti/big_rt/lib/processing/state_manager/composite/factory.h>
#include <ads/bsyeti/big_rt/lib/supplier/supplier.h>

#include <ads/bsyeti/libs/profiling/safe_stats/safe_stats.h>
#include <ads/bsyeti/libs/profiling/solomon/exporter.h>
#include <ads/bsyeti/libs/tpqlib_bridges/tpqlib_bridges.h>

#include <ads/bsyeti/libs/ytex/client/cache.h>
#include <ads/bsyeti/libs/ytex/client/proto/config.pb.h>
#include <ads/bsyeti/libs/ytex/common/await.h>
#include <ads/bsyeti/libs/ytex/http/server.h>
#include <ads/bsyeti/libs/ytex/http/std_handlers.h>
#include <ads/bsyeti/libs/ytex/program/program.h>

#include <build/scripts/c_templates/svnversion.h>

#include <yt/yt/core/misc/shutdown.h>
#include <yt/yt/core/net/address.h>
#include <yt/yt/core/net/config.h>

#include <util/digest/murmur.h>
#include <util/generic/xrange.h>
#include <util/string/join.h>
#include <util/thread/pool.h>

#include <crypta/siberia/bin/core/lib/configs/proto/logbroker_config.pb.h>
#include <crypta/lib/native/tvm/proto/tvm_config.pb.h>
#include <crypta/lib/native/log/wrappers/pqlib/pq_spdlog_wrapper.h>

#include <crypta/lib/native/pqlib/producer.h>
#include <crypta/lib/native/pqlib/pqlib.h>
#include <crypta/lib/native/tvm/create_tvm_client.h>

using namespace NBigRT;

void StaticSensorsWrite(NSFStats::TSolomonContext ctx) {
    using TLastMetric = NSFStats::TLastMetric<ui64>;
    NSFStats::TSolomonContext sctx{ctx.Detached(), {{"place", "/"}}};
    sctx.Get<TLastMetric>("revision").Set(GetProgramSvnRevision());
    sctx.Get<TLastMetric>("patch").Set(GetArcadiaPatchNumber());
    sctx.Get<TLastMetric>("build_time").Set(GetProgramBuildTimestamp());
    sctx.Get<TLastMetric>("start_time").Set(TInstant::Now().Seconds());
}

void RunMichurinService(const TMichurinConfig& config, const NYT::TCancelableContextPtr& cancelableContext) {
    auto solomonExporter = NBSYeti::NProfiling::CreateSolomonExporter(config.GetSolomonExporter());
    solomonExporter->Start();

    auto httpServer = NYTEx::NHttp::CreateServer(config.GetHttpServer());
    NYTEx::NHttp::AddStandardHandlers(
        httpServer, cancelableContext, config,
        BIND([solomonExporter] {
            return NYT::TSharedRef::FromString(NBSYeti::NProfiling::GetSensors(solomonExporter));
        }));
    httpServer->Start();
    INFO_LOG << "Started http server" << Endl;

    auto rootCtx = NBigRT::MakeSolomonContext({});
    StaticSensorsWrite(rootCtx);

    const auto& michurinSystemConfig = config.GetMichurinSystemConfig();
    const auto& cryptaIdSystemConfig = config.GetCryptaIdSystemConfig();

    NSFStats::TSolomonContext michurinCtx{
        rootCtx, {{"system", michurinSystemConfig.GetName()}}};
    NSFStats::TSolomonContext cryptaIdCtx{
        rootCtx, {{"system", cryptaIdSystemConfig.GetName()}}};

    auto ytClients = NYTEx::NRpc::CreateClientsCache(config.GetYtRpc());

    auto michurinInflightLimiter = CreateProfiledInflightLimiter(michurinCtx, michurinSystemConfig.GetMaxInflightBytes());
    auto cryptaIdInflightLimiter = CreateProfiledInflightLimiter(cryptaIdCtx, cryptaIdSystemConfig.GetMaxInflightBytes());

    auto multipleMichurinChildFactory = NYT::New<
        NMichurin::TMultipleMichurinStateDescriptor::TFactory::TChildFactory>(
        michurinSystemConfig.GetStateManagerConfig(),
        NMichurin::TMichurinStateTableOperator({
            .Table = michurinSystemConfig.GetMichurinStateProcessorConfig().GetStateTable(),
            .KeyColumn = "Id",
            .ValueColumn = "State",
            .CodecColumn = "Codec",
            .Codec = "zstd_6",
        }),
        michurinCtx);

    auto multipleMichurinFactory = NYT::New<
        NMichurin::TMultipleMichurinStateDescriptor::TFactory>(
        multipleMichurinChildFactory);

    auto compositeFactory = NYT::New<
        TCompositeStateManagerFactory>();

    NMichurin::TMichurinDescriptors descriptors{
        .MultipleMichurinDescriptor = compositeFactory->Add(
            std::move(multipleMichurinFactory)),
    };

    auto cryptaIdStateManagerFactory{
        NYT::New<TSimpleProtoStateManagerFactory<NCrypta::NIdentifiersProto::TGenericID>>(
            cryptaIdSystemConfig.GetStateManagerConfig(),
            TSimpleProtoStateTableOperator({
                .Table = cryptaIdSystemConfig.GetCryptaIdStateProcessorConfig().GetStateTable(),
                .KeyColumn = "Id",
                .ValueColumn = "CryptaId",
            }),
            cryptaIdCtx)};

    INFO_LOG << "Creating lb producer" << Endl;
    // init lb producer
    const auto& lbConfig = config.GetLogbrokerConfig();
    auto pqLib = NCrypta::NPQ::NPQLib::Create(lbConfig.GetPqLib());
    auto pqLibLogger = MakeIntrusive<NCrypta::NLog::TPqSpdlogWrapper>(lbConfig.GetPqlibLogName());

    NMichurin::TProducerPtr rewindProducer = nullptr;
    if (lbConfig.GetEnableRewinds()) {
        auto tvmConfig = config.GetTvmApiConfig();
        tvmConfig.SetSecret(GetEnv("CRYPTA_TVM_SECRET"));

        const auto tvmClient = std::make_shared<NTvmAuth::TTvmClient>(CreateTvmClient(tvmConfig));
        const auto credentialsProvider = NPersQueue::CreateTVMCredentialsProvider(tvmClient, pqLibLogger);
        INFO_LOG << "Created TVM credentials provider with client" << Endl;

        rewindProducer = MakeAtomicShared<NCrypta::NPQ::TProducer>(pqLib, lbConfig.GetRewindProducerConfig(), credentialsProvider, TStats::TSettings());
        INFO_LOG << "Created lb rewind producer" << Endl;
    } else {
        WARNING_LOG << "No rewind topic. LogBroker rewinds will be disabled" << Endl;
    }

    auto michurinConsSystem = CreateConsumingSystem({
        .Config = michurinSystemConfig.GetConsumingSystem(),
        .SuppliersProvider = CreateSupplierFactoriesProvider({.ConfigsRepeated = michurinSystemConfig.GetSuppliers()}),
        .YtClients = ytClients,
        .ShardsProcessor = [=, &descriptors](TConsumingSystem::IConsumer& consumer) {
            NMichurin::TMichurinProcessor{
                NMichurin::TMichurinProcessorBase::TConstructionArgs{
                    consumer,
                    michurinSystemConfig.GetStatefulShardProcessorConfig(),
                    compositeFactory,
                    NYTEx::CreateTransactionKeeper(ytClients, TDuration::Zero()),
                    michurinInflightLimiter,
                    michurinCtx,
                    NYT::GetCurrentInvoker()},
                michurinSystemConfig.GetMichurinStateProcessorConfig(),
                descriptors,
                rewindProducer}
                .Run();
        },
        .SensorsContext = michurinCtx,
    });

    auto cryptaIdConsSystem = CreateConsumingSystem({
        .Config = cryptaIdSystemConfig.GetConsumingSystem(),
        .SuppliersProvider = CreateSupplierFactoriesProvider({.ConfigsRepeated = cryptaIdSystemConfig.GetSuppliers()}),
        .YtClients = ytClients,
        .ShardsProcessor = [=](TConsumingSystem::IConsumer& consumer) {
            NMichurin::TCryptaIdProcessor{
                NMichurin::TCryptaIdProcessorBase::TConstructionArgs{
                    consumer,
                    cryptaIdSystemConfig.GetStatefulShardProcessorConfig(),
                    cryptaIdStateManagerFactory,
                    NYTEx::CreateTransactionKeeper(ytClients, TDuration::Zero()),
                    cryptaIdInflightLimiter,
                    cryptaIdCtx,
                    NYT::GetCurrentInvoker()},
                cryptaIdSystemConfig.GetCryptaIdStateProcessorConfig()}
                .Run();
        },
        .SensorsContext = cryptaIdCtx,
    });

    michurinConsSystem->Run();
    cryptaIdConsSystem->Run();
    NYTEx::WaitFor(cancelableContext);
    INFO_LOG << "Stopping consuming systems\n";
    NYT::NConcurrency::WaitUntilSet(michurinConsSystem->Stop());
    NYT::NConcurrency::WaitUntilSet(cryptaIdConsSystem->Stop());
}

class TMichurin: public NYTEx::TProgram<TMichurinConfig> {
public:
    TMichurin()
        : NYTEx::TProgram<TMichurinConfig>("michurin")
    {
    }

    int DoRun(const NYT::TCancelableContextPtr& context) override {
        RunMichurinService(Config(), context);
        return 0;
    }
};

int main(int argc, const char** argv) {
    return NYTEx::RunProgram<TMichurin>(argc, argv);
}
