#include <library/cpp/testing/unittest/registar.h>

#include <library/cpp/protobuf/json/proto2json.h>

#include <crypta/graph/rt/sklejka/michurin/graph_handler/graph_handler.h>
#include <crypta/graph/rt/sklejka/michurin/proto/state.pb.h>

#include <crypta/graph/rt/events/events.h>
#include <crypta/graph/rt/events/proto/event.pb.h>
#include <crypta/graph/rt/events/proto/soup.pb.h>

#include <crypta/lib/native/identifiers/lib/generic.h>

#include <util/generic/string.h>
#include <util/generic/hash.h>

using namespace NCrypta;
using namespace NMichurin;
using TGenericID = NIdentifiers::TGenericID;

using ESourceType = NSoup::NSourceType::ESourceType;
using ELogSourceType = NSoup::NLogSource::ELogSourceType;

NGraphEngine::TEdgeBetween& fill_edge(NGraphEngine::TEdgeBetween& e,
                                      const TGenericID& v1,
                                      const TGenericID& v2,
                                      ESourceType source_type,
                                      ELogSourceType log_source,
                                      ui32 seen_count = 0) {
    e.MutableVertex1()->CopyFrom(v1.ToProto());
    e.MutableVertex2()->CopyFrom(v2.ToProto());
    e.SetSourceType(source_type);
    e.SetLogSource(log_source);
    e.SetSeenCount(seen_count);
    return e;
}

struct TEdgeData {
    ui32 Vertex1 = 0;
    ui32 Vertex2 = 0;
    ESourceType SourceType;
    ELogSourceType LogSource;
    ui32 TimeStamp = 0;
    bool operator==(const TEdgeData& other) const {
        return (Vertex1 == other.Vertex1 &&
                Vertex2 == other.Vertex2 &&
                SourceType == other.SourceType &&
                LogSource == other.LogSource &&
                TimeStamp == other.TimeStamp);
    }
};

#define CHECK_EDGES(graph, ...)                     \
    do {                                            \
        TVector<TEdgeData> etalon(__VA_ARGS__);     \
        TVector<TEdgeData> computed;                \
        for (const auto& e : graph.GetEdges()) {    \
            computed.push_back({e.GetVertex1(),     \
                                e.GetVertex2(),     \
                                e.GetSourceType(),  \
                                e.GetLogSource(),   \
                                e.GetTimeStamp()}); \
        }                                           \
        UNIT_ASSERT_VALUES_EQUAL(computed, etalon); \
    } while (false)

#define CHECK_VERTICES(graph, ...)                  \
    do {                                            \
        TVector<TGenericID> etalon(__VA_ARGS__);    \
        TVector<TGenericID> computed;               \
        for (const auto& v : graph.GetVertices()) { \
            computed.push_back(TGenericID(v));      \
        }                                           \
        UNIT_ASSERT_VALUES_EQUAL(computed, etalon); \
    } while (false)

Y_UNIT_TEST_SUITE(TestGraphHandling) {
    Y_UNIT_TEST(AddingEdges) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "1"),
                                  TGenericID("mm_device_id", "2"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);

        CHECK_VERTICES(graph, {TGenericID("puid", "1"), TGenericID("mm_device_id", "2")});
        CHECK_EDGES(graph, {{0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1}});

        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "1"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        2);
        CHECK_VERTICES(graph, {TGenericID("puid", "1"), TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
                               {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 2},
                           });

        handler.AddEdge(fill_edge(e,
                                  TGenericID("mm_device_id", "2"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        3);
        CHECK_VERTICES(graph, {TGenericID("puid", "1"), TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
                               {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 2},
                               {1, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 3},
                           });
    } // Y_UNIT_TEST(AddingEdges)

    Y_UNIT_TEST(AddingEdgeMultipleTimes) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        for (int i = 0; i < 10; ++i) {
            handler.AddEdge(fill_edge(e,
                                      TGenericID("puid", "1"),
                                      TGenericID("mm_device_id", "2"),
                                      ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                            i);
            handler.AddEdge(fill_edge(e,
                                      TGenericID("puid", "1"),
                                      TGenericID("mm_device_id", "3"),
                                      ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                            i);
            CHECK_VERTICES(graph, {TGenericID("puid", "1"), TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3")});
            CHECK_EDGES(graph, {
                                   {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, static_cast<ui32>(i)},
                                   {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, static_cast<ui32>(i)},
                               });
        }
    } // Y_UNIT_TEST(AddingEdgeMultipleTimes)

    Y_UNIT_TEST(AddingEdgeMultipleTimesDeleteImmediately) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        for (int i = 0; i < 3; ++i) {
            handler.AddEdge(fill_edge(e,
                                      TGenericID("puid", "1"),
                                      TGenericID("mm_device_id", "2"),
                                      ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                            1);
        }
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "1"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        10);
        CHECK_VERTICES(graph, {TGenericID("puid", "1"), TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, static_cast<ui32>(1)},
                               {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, static_cast<ui32>(10)},
                           });
        const auto& dropped = handler.LimitEdges(handler.size() - 1);
        UNIT_ASSERT_VALUES_EQUAL(dropped, THashSet<TGenericID>{TGenericID("mm_device_id", "2")});
        // would be 0 2 if vertex 2 not deleted correctly
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, static_cast<ui32>(10)},
                           });
        // whould be 1, 2, 3  if vertex 2 not deleted correctly
        CHECK_VERTICES(graph, {TGenericID("puid", "1"), TGenericID("mm_device_id", "3")});

    } // Y_UNIT_TEST(AddingEdgeMultipleTimesDeleteImmediately)

    Y_UNIT_TEST(AddingEdgeMultipleTimesDifferentHandlers) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "1"),
                                  TGenericID("mm_device_id", "2"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        10);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "1"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        10);
        CHECK_VERTICES(graph, {TGenericID("puid", "1"), TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 10},
                               {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 10},
                           });

        TGraphHandler other_handler(&graph);
        other_handler.AddEdge(fill_edge(e,
                                        TGenericID("puid", "1"),
                                        TGenericID("mm_device_id", "2"),
                                        ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                              10);
        other_handler.AddEdge(fill_edge(e,
                                        TGenericID("puid", "1"),
                                        TGenericID("mm_device_id", "3"),
                                        ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                              10);

        CHECK_VERTICES(graph, {TGenericID("puid", "1"), TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 10},
                               {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 10},
                           });
    } // Y_UNIT_TEST(AddingEdgeMultipleTimesDifferentHandlers)

    Y_UNIT_TEST(AddingEdgeSameVerticesDifferentEdges) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "1"),
                                  TGenericID("mm_device_id", "2"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "1"),
                                  TGenericID("mm_device_id", "2"),
                                  ESourceType::PAGE_TITLE, ELogSourceType::ACCESS_LOG),
                        1);
        CHECK_VERTICES(graph, {TGenericID("puid", "1"), TGenericID("mm_device_id", "2")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
                               {0, 1, ESourceType::PAGE_TITLE, ELogSourceType::ACCESS_LOG, 1},
                           });
    } // Y_UNIT_TEST(AddingEdgeSameVerticesDifferentEdges)

    Y_UNIT_TEST(RemoveAddRemoveEdges) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);
        NGraphEngine::TEdgeBetween e;

        handler.AddEdge(fill_edge(e,
                                  TGenericID("mm_device_id", "1"),
                                  TGenericID("mm_device_id", "2"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);

        handler.AddEdge(fill_edge(e,
                                  TGenericID("mm_device_id", "2"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        2);

        handler.AddEdge(fill_edge(e,
                                  TGenericID("mm_device_id", "3"),
                                  TGenericID("mm_device_id", "4"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        3);

        handler.AddEdge(fill_edge(e,
                                  TGenericID("mm_device_id", "4"),
                                  TGenericID("mm_device_id", "1"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        4);

        CHECK_VERTICES(graph, {TGenericID("mm_device_id", "1"), TGenericID("mm_device_id", "2"),
                               TGenericID("mm_device_id", "3"), TGenericID("mm_device_id", "4")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
                               {1, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 2},
                               {2, 3, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 3},
                               {3, 0, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 4},
                           });

        handler.LimitEdges(handler.size() - 1);
        CHECK_VERTICES(graph, {TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3"),
                               TGenericID("mm_device_id", "4"), TGenericID("mm_device_id", "1")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 2},
                               {1, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 3},
                               {2, 3, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 4},
                           });

        handler.LimitEdges(handler.size() - 1);
        CHECK_VERTICES(graph, {TGenericID("mm_device_id", "3"), TGenericID("mm_device_id", "4"),
                               TGenericID("mm_device_id", "1")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 3},
                               {1, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 4},
                           });

        handler.AddEdge(fill_edge(e,
                                  TGenericID("mm_device_id", "2"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        5);
        CHECK_VERTICES(graph, {TGenericID("mm_device_id", "3"), TGenericID("mm_device_id", "4"),
                               TGenericID("mm_device_id", "1"), TGenericID("mm_device_id", "2")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 3},
                               {1, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 4},
                               {3, 0, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 5},
                           });

        // Now including a swapped vertex
        handler.AddEdge(fill_edge(e,
                                  TGenericID("mm_device_id", "4"),
                                  TGenericID("mm_device_id", "2"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        6);

        CHECK_VERTICES(graph, {TGenericID("mm_device_id", "3"), TGenericID("mm_device_id", "4"),
                               TGenericID("mm_device_id", "1"), TGenericID("mm_device_id", "2")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 3},
                               {1, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 4},
                               {3, 0, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 5},
                               {1, 3, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 6},
                           });
    } // Y_UNIT_TEST(RemoveAddRemoveEdges)

    Y_UNIT_TEST(LimitEdges) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);
        NGraphEngine::TEdgeBetween e;

        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "1"),
                                  TGenericID("mm_device_id", "2"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);

        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "1"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        2);

        handler.AddEdge(fill_edge(e,
                                  TGenericID("mm_device_id", "2"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        3);

        CHECK_VERTICES(graph, {TGenericID("puid", "1"),
                               TGenericID("mm_device_id", "2"),
                               TGenericID("mm_device_id", "3")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
                               {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 2},
                               {1, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 3},
                           });

        UNIT_ASSERT_VALUES_EQUAL(handler.size(), 3);
        {
            const auto& dropped = handler.LimitEdges(handler.size() - 1);
            UNIT_ASSERT_VALUES_EQUAL(dropped, THashSet<TGenericID>{});
        }
        UNIT_ASSERT_VALUES_EQUAL(handler.size(), 2);
        CHECK_VERTICES(graph, {TGenericID("puid", "1"),
                               TGenericID("mm_device_id", "3"),
                               TGenericID("mm_device_id", "2")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 2},
                               {2, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 3},
                           });

        {
            const auto& dropped = handler.LimitEdges(handler.size() - 1);
            UNIT_ASSERT_VALUES_EQUAL(dropped, THashSet<TGenericID>{TGenericID("puid", "1")});
        }
        CHECK_VERTICES(graph, {TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3")});
        CHECK_EDGES(graph, {
                               {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 3},
                           });

        {
            const auto& dropped = handler.LimitEdges(handler.size() - 1);
            UNIT_ASSERT_VALUES_EQUAL(dropped, (THashSet<TGenericID>{TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3")}));
        }
        CHECK_VERTICES(graph, {});
        CHECK_EDGES(graph, {});
    } // Y_UNIT_TEST(LimitEdges)

    Y_UNIT_TEST(ToSoupSeenCount) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);
        NGraphEngine::TEdgeBetween e;

        int puid1Count{5}, puid2Count{10}, puid3Count{15};
        for (int i = 0; i < puid1Count; ++i) {
            handler.AddEdge(fill_edge(e,
                                      TGenericID("puid", "1"),
                                      TGenericID("mm_device_id", "1"),
                                      ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                            i);
        }
        for (int i = 0; i < puid2Count; ++i) {
            handler.AddEdge(fill_edge(e,
                                      TGenericID("puid", "2"),
                                      TGenericID("mm_device_id", "1"),
                                      ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                            i);
        }
        for (int i = 0; i < 2; ++i) {
            handler.AddEdge(fill_edge(e,
                                      TGenericID("puid", "3"),
                                      TGenericID("mm_device_id", "1"),
                                      ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                            i);
        }
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "3"),
                                  TGenericID("mm_device_id", "1"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG,
                                  puid3Count),
                        1);
        UNIT_ASSERT_VALUES_EQUAL(handler.ToSoup().size(), 3);
        UNIT_ASSERT_VALUES_EQUAL(handler.ToSoup()[0].GetEdge().GetSeenCount(), puid1Count);
        UNIT_ASSERT_VALUES_EQUAL(handler.ToSoup()[1].GetEdge().GetSeenCount(), puid2Count);
        UNIT_ASSERT_VALUES_EQUAL(handler.ToSoup()[2].GetEdge().GetSeenCount(), puid3Count);

        int puid4Count{10};
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "4"),
                                  TGenericID("mm_device_id", "1"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG,
                                  puid4Count),
                        1);
        UNIT_ASSERT_VALUES_EQUAL(handler.ToSoup()[3].GetEdge().GetSeenCount(), puid4Count);
    } // Y_UNIT_TEST(ToSoupSeenCount)

    Y_UNIT_TEST(TrivialSplitGraphNotStrong) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "2"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "4"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        const auto toRewind = handler.Split(false);
        UNIT_ASSERT_VALUES_EQUAL(toRewind.size(), 1);
        for (const auto& [cid, events] : toRewind) {
            UNIT_ASSERT_VALUES_EQUAL(events.size(), 1);
        }

        CHECK_VERTICES(graph, {TGenericID("puid", "111"), TGenericID("mm_device_id", "2"), TGenericID("mm_device_id", "3")});
        CHECK_EDGES(graph, {
            {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
            {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
        });
    } // Y_UNIT_TEST(TrivialSplitGraphNotStrong)

    Y_UNIT_TEST(DontSplitConnectedGraphNotStrong) {
        // this test should crash because split-algorithm will delete at least one edge
        // if edges are not isStrong
        UNIT_ASSERT_TEST_FAILS([] {
            NGraphEngine::TGraph graph;
            TGraphHandler handler(&graph);

            NGraphEngine::TEdgeBetween e;
            handler.AddEdge(fill_edge(e,
                                      TGenericID("puid", "111"),
                                      TGenericID("mm_device_id", "1"),
                                      ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                            1);
            handler.AddEdge(fill_edge(e,
                                      TGenericID("puid", "222"),
                                      TGenericID("mm_device_id", "1"),
                                      ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                            1);
            handler.AddEdge(fill_edge(e,
                                      TGenericID("puid", "222"),
                                      TGenericID("mm_device_id", "2"),
                                      ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                            1);
            const auto toRewind = handler.Split(false);

            CHECK_VERTICES(graph, {
                TGenericID("puid", "111"), TGenericID("puid", "222"), TGenericID("mm_device_id", "1"), TGenericID(
                        "mm_device_id", "2")
            });
            CHECK_EDGES(graph, {
                { 0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1 },
                { 1, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1 },
                { 2, 3, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1 },
            });
        }());
    } // Y_UNIT_TEST(DontSplitConnectedGraphNotStrong)

    Y_UNIT_TEST(SplitKeepBiggestComponentNotStrong) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "1"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "2"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "333"),
                                  TGenericID("mm_device_id", "3"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "444"),
                                  TGenericID("mm_device_id", "4"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "444"),
                                  TGenericID("mm_device_id", "5"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        const auto toRewind = handler.Split(false);
        UNIT_ASSERT_VALUES_EQUAL(toRewind.size(), 3);
        for (const auto& [cid, events] : toRewind) {
            UNIT_ASSERT_VALUES_EQUAL(events.size(), 1);
        }
        CHECK_VERTICES(graph, {TGenericID("puid", "444"), TGenericID("mm_device_id", "4"), TGenericID("mm_device_id", "5")});
        CHECK_EDGES(graph, {
            {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
            {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
        });
    } // Y_UNIT_TEST(SplitKeepBiggestComponentNotStrong)

    Y_UNIT_TEST(SplitConnectedFlowerGraphsNotStrong) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "11"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "12"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "13"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "14"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "14"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "21"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "22"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "23"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "333"),
                                  TGenericID("mm_device_id", "23"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "333"),
                                  TGenericID("mm_device_id", "31"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "333"),
                                  TGenericID("mm_device_id", "32"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "333"),
                                  TGenericID("mm_device_id", "33"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        const auto toRewind = handler.Split(false);
        UNIT_ASSERT_VALUES_EQUAL(toRewind.size(), 2);
        CHECK_VERTICES(graph, {TGenericID("puid", "111"), TGenericID("mm_device_id", "11"), TGenericID("mm_device_id", "12"),
            TGenericID("mm_device_id", "13"), TGenericID("mm_device_id", "14")});
        CHECK_EDGES(graph, {
            {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
            {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
            {0, 3, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
            {0, 4, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
        });
    } // Y_UNIT_TEST(SplitConnectedFlowerGraphsNotStrong)

    Y_UNIT_TEST(DontSplitTriangleNotStrong) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "1"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "1"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("puid", "222"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        const auto toRewind = handler.Split(false);
        UNIT_ASSERT_VALUES_EQUAL(toRewind.size(), 0);
        CHECK_VERTICES(graph, {TGenericID("puid", "111"), TGenericID("mm_device_id", "1"), TGenericID("puid", "222")});
        CHECK_EDGES(graph, {
            {0, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
            {2, 1, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
            {0, 2, ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG, 1},
        });
    } // Y_UNIT_TEST(DontSplitTriangleNotStrong)

    Y_UNIT_TEST(SplitConnectedFlowerGraphsIsStrong) {
        NGraphEngine::TGraph graph;
        TGraphHandler handler(&graph);

        NGraphEngine::TEdgeBetween e;
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "11"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "12"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "13"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "111"),
                                  TGenericID("mm_device_id", "14"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "14"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "21"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "22"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "222"),
                                  TGenericID("mm_device_id", "23"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "333"),
                                  TGenericID("mm_device_id", "23"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "333"),
                                  TGenericID("mm_device_id", "31"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "333"),
                                  TGenericID("mm_device_id", "32"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        handler.AddEdge(fill_edge(e,
                                  TGenericID("puid", "333"),
                                  TGenericID("mm_device_id", "33"),
                                  ESourceType::ADV_BLOCK, ELogSourceType::WATCH_LOG),
                        1);
        const auto toRewind = handler.Split(true);
        UNIT_ASSERT_VALUES_EQUAL(toRewind.size(), 0);
        UNIT_ASSERT_VALUES_EQUAL(graph.GetVertices().size(), 13);
        UNIT_ASSERT_VALUES_EQUAL(graph.GetEdges().size(), 12);
    } // Y_UNIT_TEST(SplitConnectedFlowerGraphsIsStrong)
} // Y_UNIT_TEST_SUITE(TestGraphHandling)

template <>
void Out<TGenericID>(IOutputStream& out, const TGenericID& id) {
    out << NProtobufJson::Proto2Json(id.ToProto());
}

template <>
void Out<TEdgeData>(IOutputStream& out, const TEdgeData& edge) {
    out << "{" << edge.Vertex1 << " " << edge.Vertex2 << " " << static_cast<i32>(edge.SourceType) << " " << static_cast<i32>(edge.LogSource) << " " << edge.TimeStamp << "}";
}

template <>
void Out<TEdgeDataTuple>(IOutputStream& out, const TEdgeDataTuple& edge) {
    out << "{" << std::get<0>(edge) << " " << std::get<1>(edge) << " " << static_cast<i32>(std::get<2>(edge)) << " " << static_cast<i32>(std::get<3>(edge)) << "}";
}

template <>
void Out<TVertexInfo>(IOutputStream& out, const TVertexInfo& vertex) {
    out << "{" << vertex.index << ", " << vertex.refCount << "}";
}
