#include <crypta/graph/rt/events/proto/vulture.pb.h>
#include <crypta/graph/rt/lib/sensors/time_shift.h>
#include <crypta/graph/rt/sklejka/lib/utils.h>
#include <crypta/graph/rt/sklejka/michurin/logger/logger.h>
#include <crypta/lib/native/identifiers/lib/ads/ads.h>
#include <crypta/lib/proto/identifiers/identifiers.pb.h>

#include <ads/bsyeti/big_rt/lib/processing/state_manager/base/helpers.h>
#include <ads/bsyeti/big_rt/lib/queue/qyt/queue.h>
#include <ads/bsyeti/libs/experiments/user_ids/user_ids.h>
#include <ads/bsyeti/libs/proto_utils/profile_desc.h>

#include "michurin_processor.h"

namespace NMichurin {
    using TEventMessage = NCrypta::NEvent::TEventMessage;
    using TVultureEvent = NCrypta::NEvent::TVultureEvent;

    TMichurinProcessor::TMichurinProcessor(
        TMichurinProcessorBase::TConstructionArgs spArgs,
        const TMichurinStateProcessorConfig& config,
        TMichurinDescriptors descriptors,
        TProducerPtr rewindProducer)
        : TMichurinProcessorBase(std::move(spArgs))
        , Config(config)
        , Descriptors(std::move(descriptors))
        , RewindProducer(rewindProducer)
        , CidPacker(Config.GetReshardingModule(), {.MaxMessagesCount = Config.GetCryptaIdMaxMessagesCount()})
        , BruPacker(Config.GetBrusilov())
        , RewindPacker(1, {.MaxMessagesCount = Config.GetRewindMaxMessagesCount()})
        , Debounce("deb", Config.GetDebounceCfg())
              {};

    void TMichurinProcessor::Run() {
        YT_LOG_DEBUG("Preparing for a new run of %v shard", Shard);

        GetWhitelistRules();
        TMichurinProcessorBase::Run();
    }

    NYT::TFuture<TMichurinProcessor::TPrepareForAsyncWriteResult> TMichurinProcessor::PrepareForAsyncWrite() {
        auto cidDataForWrite = MakeAtomicShared<TVector<TYtQueue::TWriteRow>>(std::move(CidPacker.Finish()));
        CidPacker.Clear();

        auto bruDataForWrite = MakeAtomicShared<TVector<TYtQueue::TWriteRow>>(std::move(BruPacker.Finish()));
        BruPacker.Clear();

        auto rewindDataForWrite = MakeAtomicShared<THashMap<ui64, TVector<TString>>>();
        (*rewindDataForWrite)[TYtQueue::AnyShard] = Rewinds;
        Rewinds.clear();

        return NYT::MakeFuture<TPrepareForAsyncWriteResult>(
            {.AsyncWriter = [Config = this->Config, cidDataForWrite, bruDataForWrite, rewindDataForWrite](NYT::NApi::ITransactionPtr tx) {
                TYtQueue{Config.GetRewindQueue(), tx->GetClient()}.Write(tx, *rewindDataForWrite);
                TYtQueue{Config.GetCryptaIdQueue(), tx->GetClient()}.Write(tx, *cidDataForWrite);
                if (Config.GetBrusilov().GetEnabled()) {
                    TYtQueue{Config.GetBrusilov().GetQueue(), tx->GetClient()}.Write(tx, *bruDataForWrite);
                }
            }});
    }

    void TMichurinProcessor::GetWhitelistRules() {
        if (Config.GetLogWhitelistTable().empty()) {
            return;
        }
        auto client = TransactionKeeper->GetClient(Cluster);
        auto query = TStringBuilder{} << "* FROM [" << Config.GetLogWhitelistTable() << "]";
        NYT::NApi::TSelectRowsResult selectResult;
        try {
            selectResult = NYT::NConcurrency::WaitFor(client->SelectRows(query)).ValueOrThrow();
        } catch (...) {
            YT_LOG_ERROR("Failed to get log whitelist rules");
            return;
        }

        LogCryptaids.clear();
        LogShards.clear();
        LogGids.clear();

        auto schema = selectResult.Rowset->GetSchema();
        i32 ruleColumn = schema->GetColumnIndexOrThrow("Rule");

        const auto& rows = selectResult.Rowset->GetRows();
        for (const auto& row : rows) {
            TLogWhitelistRule rule;
            google::protobuf::TextFormat::ParseFromString(
                row[ruleColumn].AsString(), &rule);
            for (const auto& cid : rule.GetCryptaids()) {
                LogCryptaids.emplace(std::move(cid));
            }
            for (const auto& shard : rule.GetShards()) {
                LogShards.emplace(std::move(shard));
            }
            for (const auto& vertex : rule.GetVertices()) {
                LogGids.emplace(std::move(TGenericID(vertex)));
            }
        }
    }

    TMichurinProcessor::TGroupedChunk TMichurinProcessor::PrepareGroupedChunk(TString dataSource,
                                                                              TMichurinProcessorBase::TManager& stateManager,
                                                                              TMessageBatch data) {
        YT_LOG_DEBUG("Parsed message of type %v", dataSource);
        auto ctx = GetStatsContext({});

        ui64 messagesDebounceSkipped{0}, soupMessages{0}, totalMessages{0};

        TGroupedChunk result;
        for (auto& message : data.Messages) {
            message.Unpack();

            TEventMessage eventMessage{};
            TStringBuf skip;
            for (NFraming::TUnpacker unpacker(message.Data); unpacker.NextFrame(eventMessage, skip);) {
                ++totalMessages;
                const auto body{NCrypta::NEvent::UnpackAny(eventMessage)};
                const auto messageTypeName{NCrypta::NEvent::EMessageType_Name(eventMessage.GetType())};
                {
                    auto mctx{GetStatsContext({{"type", messageTypeName}})};
                    NCrypta::SetMessageLag(mctx, eventMessage);
                }

                if (eventMessage.GetType() == static_cast<ui64>(NCrypta::NEvent::EMessageType::SOUP)) {
                    ++soupMessages;
                    const auto& soupEvent{static_cast<TSoupEvent&>(*body)};
                    {
                        const auto messageTypeSource{NCrypta::NEvent::ESource_Name(soupEvent.GetSource())};
                        auto mctx{GetStatsContext({{"type", messageTypeName}, {"source", messageTypeSource}})};
                        NCrypta::SetMessageLag(mctx, eventMessage);
                    }
                    // NOTE(k-zaitsev): Only debounce non-merge/non-rewound events. i.e. pure events from lb queues, not rewinds/merges
                    if (!(isFastForward(soupEvent) || soupEvent.GetCounter())) {
                        TString serialized;
                        Y_PROTOBUF_SUPPRESS_NODISCARD soupEvent.GetEdge().SerializeToString(&serialized);
                        if (auto duration{Debounce.Pull(serialized)}; duration.Defined()) {
                            messagesDebounceSkipped++;
                            NCrypta::SetDebounceLag(ctx, (*duration));
                            continue;
                        } else {
                            Debounce.Push(serialized);
                        }
                    }
                    PrepareSoupEvent(soupEvent, stateManager, result);
                } else if (eventMessage.GetType() == static_cast<ui64>(NCrypta::NEvent::EMessageType::MICHURIN_BOOKKEEPING)) {
                    const auto& bookkeepingEvent{static_cast<TBookkeepingEvent&>(*body)};
                    PrepareBookkeepingEvent(bookkeepingEvent, stateManager, result);
                    {
                        const auto messageTypeName{NCrypta::NEvent::TMichurinBookkeepingEvent::EBookkeepingType_Name(bookkeepingEvent.GetType())};
                        auto mctx{GetStatsContext({{"type", messageTypeName}})};
                        mctx.Get<NSFStats::TSumMetric<ui64>>("parse_bookkeeping_messages").Inc(1);
                    }
                } else {
                    YT_LOG_ERROR("Unknown event type(NEvent::EMessageType): %v "
                                 "Ignoring request", static_cast<ui64>(eventMessage.GetType()));
                }
            }
        }
        ctx.Get<NSFStats::TSumMetric<ui64>>("parse_soup_messages").Inc(soupMessages);
        ctx.Get<NSFStats::TSumMetric<ui64>>("parse_messages").Inc(totalMessages);
        ctx.Get<NSFStats::TSumMetric<ui64>>("messages_count_debounce_skipped").Inc(messagesDebounceSkipped);
        return result;
    };

    void TMichurinProcessor::PrepareBookkeepingEvent(const TBookkeepingEvent& bookkeepingEvent, TMichurinProcessorBase::TManager& stateManager, TGroupedChunk& result) {
        const auto& cryptaId = bookkeepingEvent.GetCryptaId();

        auto stateRequest = stateManager.RequestState();
        stateRequest->Set(Descriptors.MultipleMichurinDescriptor);

        auto michurinRequest = stateRequest->Get(Descriptors.MultipleMichurinDescriptor);

        michurinRequest->Add(cryptaId);

        const TChangeRequest changeRequest(bookkeepingEvent, NCrypta::NIdentifiersProto::TGenericID{TCryptaId(ToString(cryptaId)).ToProto()});
        result[std::move(stateRequest)].emplace_back(std::move(changeRequest));
    }

    void TMichurinProcessor::PrepareSoupEvent(const TSoupEvent& soupEvent, TMichurinProcessorBase::TManager& stateManager, TGroupedChunk& result) {
        const auto& cryptaid1 = soupEvent.GetCryptaId1();
        const auto& cryptaid2 = soupEvent.GetCryptaId2();

        const auto& gid1 = TGenericID(soupEvent.GetEdge().GetVertex1());
        const auto& gid2 = TGenericID(soupEvent.GetEdge().GetVertex2());
        ui32 badCount{0};
        for (const auto& gid : {gid1, gid2}) {
            if (gid.IsSignificant()) {
                continue;
            }
            auto ctx = GetStatsContext({{"vertex", gid.GetTypeString()}, {"shard", ToString(Shard)}}, soupEvent);
            if (!gid.IsValid()) {
                ctx.Get<NSFStats::TSumMetric<ui64>>("invalid_vertices").Inc(1);
            }
            if (!gid.IsSignificant()) {
                ctx.Get<NSFStats::TSumMetric<ui64>>("insignificant_vertices").Inc(1);
            }
            badCount++;
        }
        auto logLevel = GetLogLevel(soupEvent);
        if (badCount) {
            // NOTE(k-zaitsev): These should be filtered out at resharder, so we should never really be here.
            YT_LOG_WARNING(
                "Found %v insignificant vertices(s) in event: "
                "cryptaids: %v: %v(%v), %v: %v(%v). "
                "Counter: %v, Reversed: %v, Merge: %v, SeenCount: %v",
                badCount,
                cryptaid1, gid1.GetTypeString(), gid1.GetValue(),
                cryptaid2, gid2.GetTypeString(), gid2.GetValue(),
                soupEvent.GetCounter(), soupEvent.GetReversed(), soupEvent.GetMerge(), soupEvent.GetEdge().GetSeenCount());
        }

        auto stateRequest = stateManager.RequestState();
        stateRequest->Set(Descriptors.MultipleMichurinDescriptor);

        auto michurinRequest = stateRequest->Get(Descriptors.MultipleMichurinDescriptor);

        auto ctx = GetStatsContext({}, soupEvent);
        if (cryptaid1 == 0 && cryptaid2 == 0) {
            // Both cryptaids are zero: we need to generate new one (from 1st vertex)
            // and save both of them to CID state

            TString gid1Value = gid1.GetValue();
            const ui64 newCryptaId{NCrypta::GenerateCryptaId(gid1)};
            YT_LOG_EVENT(Logger, logLevel,
                         "Generated new cryptaid: %v from %v: %v. Other vertex is: %v: %v",
                         newCryptaId, gid1.GetTypeString(), gid1.GetValue(), gid2.GetTypeString(), gid2.GetValue());
            ctx.Get<NSFStats::TSumMetric<ui64>>("generate_cids").Inc(1);

            michurinRequest->Add(newCryptaId);

            const TChangeRequest changeRequest(soupEvent, TCryptaId(ToString(newCryptaId)).ToProto());
            result[std::move(stateRequest)].emplace_back(std::move(changeRequest));

        } else if (cryptaid1 == 0 || cryptaid2 == 0 || cryptaid1 == cryptaid2) {
            // Both are the same or one is zero: just request addition

            const auto& cryptaid = cryptaid1 ? cryptaid1 : cryptaid2;
            YT_LOG_EVENT(Logger, logLevel,
                         "Will add edge to %v. "
                         "cryptaids: %v: %v, %v: %v. "
                         "Counter: %v, Reversed: %v, Merge: %v, SeenCount: %v",
                         cryptaid,
                         cryptaid1, gid1.GetTypeString(), cryptaid2, gid2.GetTypeString(),
                         soupEvent.GetCounter(), soupEvent.GetReversed(), soupEvent.GetMerge(), soupEvent.GetEdge().GetSeenCount());
            ctx.Get<NSFStats::TSumMetric<ui64>>("add_requests").Inc(1);

            michurinRequest->Add(cryptaid);

            const TChangeRequest changeRequest(soupEvent, TCryptaId(ToString(cryptaid)).ToProto());
            result[std::move(stateRequest)].emplace_back(std::move(changeRequest));
        } else {
            // Both non-zero and different: we need to merge graphs and update CID state for
            // all the edges in the merged graph

            YT_LOG_EVENT(Logger, logLevel,
                         "Recieved merge request between "
                         "%v: %v and %v: %v. "
                         "Counter: %v, Reversed: %v, Merge: %v",
                         cryptaid1, gid1.GetTypeString(), cryptaid2, gid2.GetTypeString(),
                         soupEvent.GetCounter(), soupEvent.GetReversed(), soupEvent.GetMerge());
            ctx.Get<NSFStats::TSumMetric<ui64>>("merge_requests").Inc(1);

            michurinRequest->Add(cryptaid1);
            michurinRequest->Add(cryptaid2);

            const TChangeRequest changeRequest(soupEvent);
            result[std::move(stateRequest)].emplace_back(std::move(changeRequest));
        }
    };

    void TMichurinProcessor::ProcessGroupedChunk(TString dataSource, TGroupedChunk groupedRows) {
        YT_LOG_DEBUG("Processing message of type %v, got %v requests",
                     dataSource, groupedRows.size());

        for (auto& [request, rows] : groupedRows) {
            TVector<TChangeRequest> soupRows{}, bookkeepingRows{};
            for (auto& row : rows) {
                const auto& changeType = row.GetType();
                if (changeType == SOUP_CHANGE) {
                    soupRows.emplace_back(std::move(row));
                } else if (changeType == BOOKKEEPING_CHANGE) {
                    bookkeepingRows.emplace_back(std::move(row));
                } else {
                    YT_LOG_ERROR("Unknown change type(NMichurin::EChangeType): %v "
                                 "Ignoring request", static_cast<ui64>(changeType));
                }
            }

            if (!soupRows.empty()) {
                ProcessSoupRows(request, soupRows);
            }
            if (!bookkeepingRows.empty()) {
                ProcessBookkeepingRows(request, bookkeepingRows);
            }
        }
    };

    void TMichurinProcessor::ProcessSoupRows(const TGenericStateRequestPtr& stateRequest, const TVector<TChangeRequest>& rows) {
        auto michurinStates = stateRequest->Get(Descriptors.MultipleMichurinDescriptor)->Get();

        if (michurinStates.size() == 1) {
            ProcessSingleStateUpdates(stateRequest, rows);
        } else if (michurinStates.size() == 2) {
            ProcessMergeRequests(stateRequest, rows);
        } else {
            // NOTE(k-zaitsev): should never be here
            YT_LOG_ERROR("Got unexpected number of michurin states in request: %v "
                         "Expected 1 or 2. Ignoring request", michurinStates.size());
        }
    }
    void TMichurinProcessor::ProcessBookkeepingRows(const TGenericStateRequestPtr& stateRequest, const TVector<TChangeRequest>& rows) {
        const auto& eventType = rows.front().GetBookkeepingEventType();
        if (eventType == NCrypta::NEvent::TMichurinBookkeepingEvent::CID_UPDATE) {
            ProcessBookkeepingCidUpdate(stateRequest, rows);
        } else if (eventType == NCrypta::NEvent::TMichurinBookkeepingEvent::TOMBSTONE_DELETE) {
            ProcessBookkeepingTombstoneDelete(stateRequest);
        } else if (eventType == NCrypta::NEvent::TMichurinBookkeepingEvent::SPLIT) {
            ProcessBookkeepingSplit(stateRequest);
        } else {
            YT_LOG_ERROR("Got unexpected BookkeepingEvent (NEvent::TMichurinBookkeepingEvent): %v "
                         "Ignoring change request", static_cast<ui64>(eventType));
        }
    }

    TGraphHandlerPtr TMichurinProcessor::GetHandlerSafe(TMichurinState& state) {
        TGraphHandlerPtr graphHolder;
        try {
            graphHolder = MakeHolder<TGraphHandler>(state.MutableGraph());
        } catch (...) {
        }
        return graphHolder;
    }

    void TMichurinProcessor::ClearInvalidState(TMichurinState& state) {
        auto ctx = GetStatsContext({});
        ctx.Get<NSFStats::TSumMetric<ui64>>("invalid_graphs").Inc(1);

        for (const auto& vertex : state.GetGraph().GetVertices()) {
            SetCidRequest(TGenericID{vertex}, ZeroCID);
        }
        ctx.Get<NSFStats::TSumMetric<ui64>>("invalid_reset_cid").Inc(state.GetGraph().GetVertices().size());
        state.Clear();
    }

    void TMichurinProcessor::ProcessBookkeepingCidUpdate(const TGenericStateRequestPtr& stateRequest, const TVector<TChangeRequest>& rows) {
        auto ctx = GetStatsContext({});

        auto michurinRequest = stateRequest->Get(Descriptors.MultipleMichurinDescriptor)->Get().front();
        auto& state = michurinRequest->GetState();
        state.SetBookkeepingCIDUpdatedAt(TInstant::Now().Seconds());

        const auto& changeRequest = rows.front();
        const auto& cryptaId = changeRequest.GetToCid();

        for (const auto& vertex : state.GetGraph().GetVertices()) {
            SetCidRequest(TGenericID{vertex}, cryptaId);
        }

        ctx.Get<NSFStats::TSumMetric<ui64>>("bookkeeping_cid_update_vertices").Inc(state.GetGraph().GetVertices().size());
        ctx.Get<NSFStats::TSumMetric<ui64>>("bookkeeping_cid_update_states").Inc(1);
    }

    void TMichurinProcessor::ProcessBookkeepingTombstoneDelete(const TGenericStateRequestPtr& stateRequest) {
        auto ctx = GetStatsContext({});

        auto michurinRequest = stateRequest->Get(Descriptors.MultipleMichurinDescriptor)->Get().front();
        auto& state = michurinRequest->GetState();

        const auto& now = TInstant::Now().Seconds();

        if (state.GetMergedToCryptaId() != 0 &&
            (state.GetGraph().GetEdges().size() || state.GetGraph().GetVertices().size())) {
            // NOTE(k-zaitsev): Should not really happen. As soon as graph is merged no more edges should ever be added to it
            ctx.Get<NSFStats::TSumMetric<ui64>>("bookkeeping_tombstone_not_empty_graph_with_mergedto").Inc(1);
            return;
        }

        if (state.GetMergedToCryptaId() != 0 && now - state.GetTouchedAt() > Config.GetMergedStateTTL()) {
            state.Clear();
            ctx.Get<NSFStats::TSumMetric<ui64>>("bookkeeping_tombstones_deleted").Inc(1);
        }
    }

    void TMichurinProcessor::ProcessBookkeepingSplit(const TGenericStateRequestPtr& stateRequest) {
        auto ctx = GetStatsContext({});
        auto michurinRequest = stateRequest->Get(Descriptors.MultipleMichurinDescriptor)->Get().front();
        auto& state = michurinRequest->GetState();
        auto graphHandler = GetHandlerSafe(state);

        ctx.Get<NSFStats::TSumMetric<ui64>>("bookkeeping_spit_event").Inc(1);

        if (!graphHandler) {
            YT_LOG_ERROR("Found invalid graph for cryptaid: %v "
                         "Dropping it and resetting all it's edges ",
                         michurinRequest->GetStateId());
            ClearInvalidState(state);
            return;
        }

        ui64 rewoundEventsCnt{0};

        if (graphHandler->size() <= 1) {
            YT_LOG_DEBUG("Graph with no edges for cryptaid: %v", michurinRequest->GetStateId());
            ctx.Get<NSFStats::TSumMetric<ui64>>("bookkeeping_split_not_done").Inc(1);
            return;
        }

        const THashMap<ui64, TVector<TSoupEvent>>& edgesToRewind = graphHandler->Split(Config.GetForceEdgesStrongForSplit());

        for (const auto& [cid, events] : edgesToRewind) {
            const auto& toCid = TCryptaId(ToString(cid)).ToProto();
            rewoundEventsCnt += events.size();
            for (const auto& event : events) {
                SetCidRequest(TGenericID(event.GetEdge().GetVertex1()), toCid);
                SetCidRequest(TGenericID(event.GetEdge().GetVertex2()), toCid);
                Rewind(event, ESource::SPLIT, cid);
            }
        }

        ctx.Get<NSFStats::TSumMetric<ui64>>("bookkeeping_spit_rewound_events").Inc(rewoundEventsCnt);

        if (edgesToRewind.size() == 0) {
            ctx.Get<NSFStats::TSumMetric<ui64>>("bookkeeping_split_not_done").Inc(1);
        } else {
            ctx.Get<NSFStats::TSumMetric<ui64>>("bookkeeping_spit_done").Inc(1);
        }
    }

    void TMichurinProcessor::ProcessSingleStateUpdates(const TGenericStateRequestPtr& stateRequest, const TVector<TChangeRequest>& rows) {
        auto michurinRequest = stateRequest->Get(Descriptors.MultipleMichurinDescriptor)->Get().front();

        auto& state = michurinRequest->GetState();
        ui64 stateId = michurinRequest->GetStateId();
        YT_LOG_EVENT(Logger, GetLogLevel(stateId),
                     "Processing %v events for state %v",
                     rows.size(), stateId);

        auto graphHandler = GetHandlerSafe(state);
        if (!graphHandler) {
            YT_LOG_ERROR("Found invalid graph for cryptaid: %v "
                         "Dropping it and resetting all it's edges ",
                         stateId);
            ClearInvalidState(state);
            return;
        }

        if (graphHandler->GetId() == 0) {
            graphHandler->SetId(stateId);
        }

        bool edgesAdded = false;
        for (const auto& changeRequest : rows) {
            auto ctx = GetStatsContext({}, changeRequest.GetSoupEvent());
            auto logLevel = GetLogLevel(changeRequest.GetSoupEvent());
            const auto& gid1 = TGenericID(changeRequest.GetGID1());
            const auto& gid2 = TGenericID(changeRequest.GetGID2());
            const auto cryptaid1 = changeRequest.GetCryptaId1();
            const auto cryptaid2 = changeRequest.GetCryptaId2();

            if (auto movedTo = state.GetMergedToCryptaId(); movedTo != 0) {
                YT_LOG_EVENT(Logger, logLevel,
                             "Graph for %v[%v] was moved to %v "
                             "Not adding edge %v <=> %v "
                             "Counter: %v, Reversed: %v, Merge: %v, SeenCount: %v, rewinding",
                             stateId, graphHandler->size(), movedTo,
                             gid1.GetTypeString(), gid2.GetTypeString(),
                             changeRequest.GetCounter(), changeRequest.GetReversed(), changeRequest.GetMerge(), changeRequest.GetSeenCount());

                if (changeRequest.GetCounter()) {
                    const auto& toCid = TCryptaId(ToString(movedTo)).ToProto();

                    if (cryptaid1) {
                        SetCidRequest(gid1, toCid);
                        ctx.Get<NSFStats::TSumMetric<ui64>>("add_reset_cid").Inc(1);
                    }
                    if (cryptaid2) {
                        SetCidRequest(gid2, toCid);
                        ctx.Get<NSFStats::TSumMetric<ui64>>("add_reset_cid").Inc(1);
                    }
                }
                Rewind(changeRequest.GetSoupEvent());
                ctx.Get<NSFStats::TSumMetric<ui64>>("add_rewind").Inc(1);
                ctx.Get<TRewindMetric>("add_rewind_hist").Add(changeRequest.GetCounter());
                continue;
            }

            bool missingCId1 = cryptaid1 > 0 && !graphHandler->contains(gid1);
            bool missingCId2 = cryptaid2 > 0 && !graphHandler->contains(gid2);
            if (!isFastForward(changeRequest.GetSoupEvent()) && (missingCId1 || missingCId2)) {
                YT_LOG_EVENT(Logger, logLevel,
                             "Graph for %v[%v] Does not contain vertices: "
                             "%v %v"
                             "Counter: %v, Reversed: %v, Merge: %v, SeenCount: %v, rewinding",
                             stateId, graphHandler->size(),
                             (missingCId1 ? gid1.GetTypeString() : ""),
                             (missingCId2 ? gid2.GetTypeString() : ""),
                             changeRequest.GetCounter(), changeRequest.GetReversed(), changeRequest.GetMerge(), changeRequest.GetSeenCount());
                if (changeRequest.GetCounter() == Config.GetResetCIDAtCount()) {
                    if (missingCId1) {
                        SetCidRequest(gid1, ZeroCID);
                        ctx.Get<NSFStats::TSumMetric<ui64>>("add_reset_missing_cid").Inc(1);
                    }
                    if (missingCId2) {
                        SetCidRequest(gid2, ZeroCID);
                        ctx.Get<NSFStats::TSumMetric<ui64>>("add_reset_missing_cid").Inc(1);
                    }
                }
                Rewind(changeRequest.GetSoupEvent());
                ctx.Get<NSFStats::TSumMetric<ui64>>("add_rewind_missing").Inc(1);
                ctx.Get<TRewindMetric>("add_rewind_missing_hist").Add(changeRequest.GetCounter());
                continue;
            }

            YT_LOG_EVENT(Logger, logLevel,
                         "Adding edge %v <=> %v to %v[%v] "
                         "Counter: %v, Reversed: %v, Merge: %v, SeenCount: %v",
                         gid1.GetTypeString(), gid2.GetTypeString(),
                         graphHandler->GetId(), graphHandler->size(),
                         changeRequest.GetCounter(), changeRequest.GetReversed(), changeRequest.GetMerge(), changeRequest.GetSeenCount());
            bool addedEdge = graphHandler->ProcessSoupEvent(changeRequest.GetSoupEvent());
            edgesAdded |= addedEdge;

            if (cryptaid1 != changeRequest.GetToCid().GetCryptaId().GetValue()) {
                SetCidRequest(gid1, changeRequest.GetToCid());
            }
            if (cryptaid2 != changeRequest.GetToCid().GetCryptaId().GetValue()) {
                SetCidRequest(gid2, changeRequest.GetToCid());
            }
            ctx.Get<NSFStats::TSumMetric<ui64>>(addedEdge ? "added_new_edge" : "added_existing_edge").Inc(1);
            ctx.Get<NSFStats::TSumMetric<ui64>>("add_done").Inc(1);
        }

        // TODO: keep loglevel high if any of the edges had a high one
        auto logLevel = GetLogLevel(stateId);

        if (LimitEdges(graphHandler)) {
            state.SetLimitResetCount(state.GetLimitResetCount() + 1);
            YT_LOG_EVENT(Logger, logLevel,
                         "Done limiting edges for %v[%v]",
                         stateId, graphHandler->size());
        } else {
            YT_LOG_EVENT(Logger, logLevel,
                         "Did not limit edges for %v[%v]",
                         stateId, graphHandler->size());
        }

        auto now = TInstant::Now().Seconds();
        {
            // TODO(k-zaitsev): Move to config
            const auto tochedDelay{now - state.GetTouchedAt()};
            const auto tochedTreshold{Config.GetVultureTouchPeriod() + graphHandler->GetId() % (Config.GetVultureTouchPeriod() / 10)};
            const auto tochedLimiter{Config.GetVultureTouchLimiter()};

            size_t vultureSize{0};
            if ((edgesAdded && (tochedDelay >= tochedLimiter)) || (tochedDelay >= tochedTreshold)) {
                vultureSize = ConvertToVulture(graphHandler);
                state.SetTouchedAt(now);
                state.SetTouchCount(state.GetTouchCount() + 1);
                YT_LOG_EVENT(Logger, logLevel,
                             "Updated vulture for %v[%v]",
                             stateId, graphHandler->size());
            }

            if (edgesAdded && (tochedDelay >= tochedLimiter)) {
                GetStatsContext({}).Get<NSFStats::TSumMetric<ui64>>("vulture_add").Inc(1);
                GetStatsContext({{"mode", "default"}}).Get<NSFStats::TSumMetric<ui64>>("vulture_add").Inc(1);
                GetStatsContext({{"mode", "vult_size"}}).Get<NSFStats::TSumMetric<ui64>>("vulture_add").Inc(vultureSize);
                GetStatsContext({{"mode", "graph_size"}}).Get<NSFStats::TSumMetric<ui64>>("vulture_add").Inc(graphHandler->size());
            } else if (tochedDelay > tochedTreshold) {
                GetStatsContext({}).Get<NSFStats::TSumMetric<ui64>>("vulture_touch").Inc(1);
                GetStatsContext({{"mode", "default"}}).Get<NSFStats::TSumMetric<ui64>>("vulture_touch").Inc(1);
                GetStatsContext({{"mode", "vult_size"}}).Get<NSFStats::TSumMetric<ui64>>("vulture_touch").Inc(vultureSize);
                GetStatsContext({{"mode", "graph_size"}}).Get<NSFStats::TSumMetric<ui64>>("vulture_touch").Inc(graphHandler->size());
            } else {
                GetStatsContext({}).Get<NSFStats::TSumMetric<ui64>>("vulture_skip").Inc(1);
                GetStatsContext({{"mode", "default"}}).Get<NSFStats::TSumMetric<ui64>>("vulture_skip").Inc(1);
                GetStatsContext({{"mode", "vult_size"}}).Get<NSFStats::TSumMetric<ui64>>("vulture_skip").Inc(vultureSize);
                GetStatsContext({{"mode", "graph_size"}}).Get<NSFStats::TSumMetric<ui64>>("vulture_skip").Inc(graphHandler->size());
            }
        }
    }

    void TMichurinProcessor::ProcessMergeRequests(const TGenericStateRequestPtr& stateRequest, const TVector<TChangeRequest>& rows) {
        for (const auto& changeRequest : rows) {
            auto ctx = GetStatsContext({}, changeRequest.GetSoupEvent());
            auto logLevel = GetLogLevel(changeRequest.GetSoupEvent());
            auto [fromCryptaId, toCryptaId] = changeRequest.GetReversed()
                                                  ? std::make_tuple(changeRequest.GetCryptaId2(), changeRequest.GetCryptaId1())
                                                  : std::make_tuple(changeRequest.GetCryptaId1(), changeRequest.GetCryptaId2());
            auto states = stateRequest->Get(Descriptors.MultipleMichurinDescriptor)->Get();

            auto [fromIdx, toIdx] = states[0]->GetStateId() == fromCryptaId
                                        ? std::make_tuple(0, 1)
                                        : std::make_tuple(1, 0);

            auto& fromState = states[fromIdx]->GetState();
            auto& toState = states[toIdx]->GetState();
            // We never want to alter toState, since it belongs to another shard
            toState.SetSkip(true);

            auto fromGraphHandler = GetHandlerSafe(fromState);
            const auto toGraphHandler = GetHandlerSafe(toState);
            if (!fromGraphHandler || !toGraphHandler) {
                YT_LOG_ERROR("Found invalid graphs when attemprint to merge: %v to %v "
                             "Skipping.",
                             fromCryptaId, toCryptaId);
                continue;
                // TODO(k-zaitsev): clear invalid states
            }

            const auto& gid1 = TGenericID(changeRequest.GetGID1());
            const auto& gid2 = TGenericID(changeRequest.GetGID2());
            if (fromState.GetMergedToCryptaId() || toState.GetMergedToCryptaId()) {
                YT_LOG_EVENT(Logger, logLevel,
                             "Rewinding merge request, since one of the graphs has already been moved. "
                             "%v[%v] to %v"
                             "%v[%v] to %v"
                             "Edge: %v <=> %v, Counter: %v, Reversed: %v",
                             fromCryptaId, fromGraphHandler->size(), fromState.GetMergedToCryptaId(),
                             toCryptaId, toGraphHandler->size(), toState.GetMergedToCryptaId(),
                             gid1.GetTypeString(), gid2.GetTypeString(), changeRequest.GetCounter(), changeRequest.GetReversed());
                if (const auto& movedTo = fromState.GetMergedToCryptaId(); movedTo > 0) {
                    const auto& gid = changeRequest.GetReversed() ? gid2 : gid1;
                    SetCidRequest(gid, TCryptaId(ToString(movedTo)).ToProto());
                    ctx.Get<NSFStats::TSumMetric<ui64>>("merge_moved_to_reset_cid").Inc(1);
                }
                if (const auto& movedTo = toState.GetMergedToCryptaId(); movedTo > 0) {
                    const auto& gid = changeRequest.GetReversed() ? gid1 : gid2;
                    SetCidRequest(gid, TCryptaId(ToString(movedTo)).ToProto());
                    ctx.Get<NSFStats::TSumMetric<ui64>>("merge_moved_to_reset_cid").Inc(1);
                }

                // resharder should resolve this case, so we set null cryptaIds
                Rewind(changeRequest.GetSoupEvent());
                ctx.Get<NSFStats::TSumMetric<ui64>>("merge_moved_to_rewind").Inc(1);
                ctx.Get<TRewindMetric>("merge_moved_to_rewind_hist").Add(changeRequest.GetCounter());
                continue;
            }
            const auto& fromGid = changeRequest.GetReversed() ? gid2 : gid1;
            if (!fromGraphHandler->contains(fromGid)) {
                YT_LOG_EVENT(Logger, logLevel,
                             "Rewinding merge request, since fromGraph does not contain it's edge. "
                             "%v[%v] to %v[%v]"
                             "Edge: %v <=> %v, Counter: %v, Reversed: %v",
                             fromCryptaId, fromGraphHandler->size(), toCryptaId, toGraphHandler->size(),
                             gid1.GetTypeString(), gid2.GetTypeString(), changeRequest.GetCounter(), changeRequest.GetReversed());
                Rewind(changeRequest.GetSoupEvent());
                ctx.Get<NSFStats::TSumMetric<ui64>>("merge_no_contain_rewind").Inc(1);
                ctx.Get<TRewindMetric>("merge_no_contain_rewind_hist").Add(changeRequest.GetCounter());
                continue;
            }

            if (fromGraphHandler->size() > toGraphHandler->size() || (fromGraphHandler->size() == toGraphHandler->size() && fromCryptaId > toCryptaId)) {
                YT_LOG_EVENT(Logger, logLevel,
                             "Rewinding & reversing merge request. "
                             "%v[%v] to %v[%v] "
                             "Edge: %v <=> %v, Counter: %v, Reversed: %v",
                             fromCryptaId, fromGraphHandler->size(), toCryptaId, toGraphHandler->size(),
                             gid1.GetTypeString(), gid2.GetTypeString(), changeRequest.GetCounter(), changeRequest.GetReversed());
                auto event = changeRequest.GetSoupEvent();
                event.SetReversed(!event.GetReversed());
                Rewind(event);

                ctx.Get<NSFStats::TSumMetric<ui64>>("merge_reverse").Inc(1);
                continue;
            }

            auto toCid = TCryptaId(ToString(toCryptaId)).ToProto();

            YT_LOG_EVENT(Logger, logLevel,
                         "Merging %v[%v] to %v[%v]"
                         "Edge: %v <=> %v, Counter: %v, Reversed: %v",
                         fromCryptaId, fromGraphHandler->size(), toCryptaId, toGraphHandler->size(),
                         gid1.GetTypeString(), gid2.GetTypeString(), changeRequest.GetCounter(), changeRequest.GetReversed());

            fromState.SetMergedToCryptaId(toCryptaId);
            fromState.SetTouchedAt(TInstant::Now().Seconds());

            THashSet<TGenericID> movedVertices;
            ui32 mergeRewinds{0};

            fromGraphHandler->ProcessSoupEvent(changeRequest.GetSoupEvent());
            for (const auto& event : fromGraphHandler->ToSoup()) {
                const auto& movedGid1 = TGenericID(event.GetEdge().GetVertex1());
                const auto& movedGid2 = TGenericID(event.GetEdge().GetVertex2());

                YT_LOG_DEBUG("Will set moved gids from edge %v <=> %v to %v",
                             movedGid1.GetTypeString(),
                             movedGid2.GetTypeString(),
                             toCryptaId);

                movedVertices.insert(std::move(movedGid1));
                movedVertices.insert(std::move(movedGid2));

                // cryptaIds in soup events are nulls, so resharder should resolve this case
                Rewind(event, ESource::MERGE, toCryptaId);
                ++mergeRewinds;
            }

            fromGraphHandler->clear();
            for (auto& gid : movedVertices) {
                SetCidRequest(gid, toCid);
            }

            ctx.Get<NSFStats::TSumMetric<ui64>>("merge_edge_rewind").Inc(mergeRewinds);
            ctx.Get<NSFStats::TSumMetric<ui64>>("merge_done").Inc(1);
        }
    }

    void TMichurinProcessor::Rewind(const TSoupEvent& originalEvent) {
        Rewind(originalEvent, ESource::UNKNOWN, 0);
    }

    void TMichurinProcessor::Rewind(const TSoupEvent& originalEvent, ESource source, ui64 cidToSet) {
        TSoupEvent event;
        event.CopyFrom(originalEvent);
        event.SetCounter(event.GetCounter() + 1);
        event.SetCryptaId1(cidToSet);
        event.SetCryptaId2(cidToSet);
        if (source != ESource::UNKNOWN) {
            event.SetSource(source);
        }

        TString serialized;
        Y_PROTOBUF_SUPPRESS_NODISCARD event.SerializeToString(&serialized);
        serialized = Base64Encode(serialized);

        if (RewindProducer) {
            // TODO: Check for return value, flush and maybe try again. This might happen if we
            // ever set MaxInFlightBytes for producer to non-zero
            RewindProducer->TryEnqueue(serialized);
        }

        if (!Config.GetRewindQueue().empty()) {
            Rewinds.emplace_back(std::move(serialized));
        }
    }

    bool TMichurinProcessor::LimitEdges(TGraphHandlerPtr& handler) {
        if (handler->size() < Config.GetEnforceEdgeLimitAfter()) {
            return false;
        }

        auto ctx = GetStatsContext({});
        auto sizeBeforeLimit = handler->size();
        auto movedVertices = handler->LimitEdges(Config.GetEdgeLimit());
        ctx.Get<NSFStats::TSumMetric<ui64>>("limit_edges_removed").Inc(sizeBeforeLimit - handler->size());

        for (auto& vertex : movedVertices) {
            SetCidRequest(vertex, ZeroCID);
        }
        ctx.Get<NSFStats::TSumMetric<ui64>>("limit_cids_reset").Inc(movedVertices.size());
        return true;
    }

    size_t TMichurinProcessor::ConvertToVulture(const TGraphHandlerPtr& handler) {
        auto ctx{GetStatsContext({})};
        static const NBSYeti::TIdentTypeConverter enumConverter;

        size_t counter{0}, nonVultureVertexCount{0};
        for (const auto& [vertex, associated] : handler->ConvertToVulture()) {
            if (const auto keyYabsType{CryptaToAds(vertex.GetType())}; !keyYabsType.Defined()) {
                YT_LOG_DEBUG("%v is a non-vulture vertex, skipping",
                             vertex.GetTypeString());
                ++nonVultureVertexCount;
            } else {
                auto idPrefix = enumConverter.GetTupleByMainValue(*keyYabsType).IdPrefix;
                auto key = TStringBuilder{} << idPrefix << vertex.GetValue();

                BruPacker.Add(
                    NExperiments::GetBigbShardNumber(key, Config.GetBrusilov().GetReshardingModule()),
                    CreateVultEvent(key, associated)
                );
                ++counter;
            }
        }
        ctx.Get<NSFStats::TSumMetric<ui64>>("non_vulture_vertices_skip").Inc(nonVultureVertexCount);
        ctx.Get<NSFStats::TSumMetric<ui64>>("vulture_changes").Inc(1);
        ctx.Get<NSFStats::TSumMetric<ui64>>("vulture_vertices").Inc(counter);
        return counter;
    }

    TSimpleEventMessage TMichurinProcessor::CreateVultEvent(const TString& key, const TAssociatedUids& associated) {
        TSimpleEventMessage wrapped{};
        TVultureEvent event{};

        wrapped.SetTimeStamp(TInstant::Now().Seconds());
        wrapped.SetType(NCrypta::NEvent::EMessageType::VULTURE);
        wrapped.SetSource("michurin");
        {
            // todo: move to config
            event.MutableLocations()->Add(NCrypta::NEvent::EVultureLocation::VULT_DEFAULT);
            event.MutableLocations()->Add(NCrypta::NEvent::EVultureLocation::VULT_EXP);
            event.SetKeyPrefix("mi:");
        }
        event.SetId(key);
        if (associated.GetValueRecords().size()) {
            event.MutableAssociated()->CopyFrom(associated);
        }

        Y_PROTOBUF_SUPPRESS_NODISCARD event.SerializeToString(wrapped.MutableBody());
        return wrapped;
    }

    void TMichurinProcessor::SetCidRequest(const TGenericID& gid, const NCrypta::NIdentifiersProto::TGenericID& cid) {
        TCryptaIdEvent event;
        event.MutableGid()->CopyFrom(gid.ToProto());
        event.MutableCid()->CopyFrom(cid);
        CidPacker.Add(
            NExperiments::GetBigbShardNumber(gid.Serialize(), Config.GetReshardingModule()),
            event);
    }

    NSFStats::TSolomonContext TMichurinProcessor::GetStatsContext(TVector<NSFStats::TSolomonContext::TLabel> labels, const TSoupEvent& event) const {
        TStringBuilder edgeType{};
        edgeType << NCrypta::NSoup::LogSourceType(event.GetEdge().GetLogSource()).GetName() << ":"
                 << NCrypta::NSoup::SourceType(event.GetEdge().GetSourceType()).GetName() << ":"
                 << TGenericID(event.GetEdge().GetVertex1()).GetTypeString() << ":"
                 << TGenericID(event.GetEdge().GetVertex2()).GetTypeString();
        labels.push_back({"edge_type", edgeType});
        return GetStatsContext(labels);
    }

    NSFStats::TSolomonContext TMichurinProcessor::GetStatsContext(TVector<NSFStats::TSolomonContext::TLabel> labels) const {
        labels.push_back({"place", "michurin_processor"});
        return NSFStats::TSolomonContext{SensorsContext.Detached(), labels};
    }

    ELogLevel TMichurinProcessor::GetLogLevel(const ui64& cryptaId) const {
        if (LogShards.contains(Shard) || LogCryptaids.contains(cryptaId)) {
            return ELogLevel::Info;
        }
        return ELogLevel::Debug;
    }

    ELogLevel TMichurinProcessor::GetLogLevel(const TSoupEvent& event) const {
        if (LogShards.contains(Shard)) {
            return ELogLevel::Info;
        }
        if (LogCryptaids.contains(event.GetCryptaId1()) || LogCryptaids.contains(event.GetCryptaId2())) {
            return ELogLevel::Info;
        }
        if (LogGids.contains(TGenericID(event.GetEdge().GetVertex1())) || LogGids.contains(TGenericID(event.GetEdge().GetVertex2()))) {
            return ELogLevel::Info;
        }
        return ELogLevel::Debug;
    }

    bool isFastForward(const TSoupEvent& event) {
        return (event.GetMerge() ||
                event.GetSource() == ESource::MERGE ||
                event.GetSource() == ESource::SPLIT);
    }
}
