#pragma once

#include "michurin_state_manager.h"
#include <crypta/graph/rt/sklejka/michurin/logger/logger.h>

#include <library/cpp/iterator/zip.h>

namespace NMichurin {
    template <typename TState>
    TMichurinStateManager<TState>::TMichurinStateManager(
        TStateManagerConfig config,
        TMichurinStateTableOperator stateOperator,
        NSFStats::TSolomonContext sensorsContext)
        : TGenericStateManager<ui64, TState>(std::move(config), sensorsContext)
        , StateOperator(std::move(stateOperator))
    {
    }

    template <typename TState>
    TMichurinStateManager<TState>::TMichurinStateManager(
        TStateManagerConfig config,
        TMichurinStateTableOperator stateOperator,
        const NYT::NProfiling::TTagSet& tags)
        : TMichurinStateManager(std::move(config), std::move(stateOperator), MakeSolomonContext(tags))
    {
    }

    template <typename TState>
    NYT::TFuture<
        TVector<TBaseStateRequestPtr>>
    TMichurinStateManager<TState>::LoadStates(
        TVector<TBaseRequestPtr> requests,
        NYT::NApi::IClientPtr client,
        NYT::NTransactionClient::TTimestamp timestamp) {
        Y_UNUSED(timestamp);

        auto ids = TVector<TStateId>(
            Reserve(requests.size()));

        for (const auto& request : requests) {
            ids.push_back(request->GetStateId());
        }

        return StateOperator.Load(
                                std::move(client), std::move(ids))
            .Apply(BIND([stateOperator = StateOperator,
                         requests = std::move(requests)](const TVector<TString>& rows) {
                auto baseRequests = TVector<TBaseStateRequestPtr>(
                    Reserve(requests.size()));

                auto states = stateOperator.template Parse<TState>(rows);

                for (auto&& [request, state] : Zip(requests, states)) {
                    request->GetState() = std::move(state);
                    baseRequests.push_back(std::move(request));
                }

                return baseRequests;
            }));
    }

    template <typename TState>
    TBaseStateManager::TStateWriter
    TMichurinStateManager<TState>::WriteStates(
        TVector<TBaseRequestPtr> requests) {
        auto states = TVector<TState>(
            Reserve(requests.size()));

        auto ids = TVector<TStateId>(
            Reserve(requests.size()));

        //TODO: make sure sensors work around here
        //auto ctx = this->SensorsContext.Detached();
        for (auto&& request : requests) {
            const auto& state = request->GetState();
            if (state.GetSkip()) {
                //ctx.template Get<NSFStats::TSumMetric<ui64>>("skipped_by_request").Inc(1);
                YT_LOG_DEBUG("Skip requested for %v. Not writing it", request->GetStateId());
                continue;
            }
            ids.push_back(request->GetStateId());
            states.push_back(request->GetState());
        }

        auto serialized = StateOperator.Serialize(std::move(states));
        return [ids = std::move(ids),
                serialized = std::move(serialized),
                stateOperator = StateOperator](NYT::NApi::ITransactionPtr tx) {
            stateOperator.Write(std::move(tx), ids, serialized);
        };
    }

    template <typename TState>
    size_t TMichurinStateManager<TState>::SpaceUsed(const TState& state) {
        return state.SpaceUsedLong();
    }
}
