from google.protobuf import text_format

from crypta.graph.rt.sklejka.michurin.proto import log_whitelist_pb2
from crypta.lib.python.identifiers.generic_id import GenericID

import yt.wrapper as yt

import argparse
import os

DEFAULT_TABLE = "//home/crypta/testing/rtsklejka/state/whitelist"


def format_rule(rule):
    gids_desc = 'Vertices: {}'.format([
        "{}({})".format(gid.WhichOneof('identifier'), GenericID(proto=gid).value) for gid in rule.Vertices
    ]) if rule.Vertices else ''
    cid_desc = 'Cryptaids: {}'.format(rule.Cryptaids) if rule.Cryptaids else ''
    shards_desc = 'Shards: {}'.format(rule.Shards) if rule.Shards else ''
    return ', '.join([part for part in [gids_desc, cid_desc, shards_desc] if part])


def list_rows(client, path):
    rows = list(client.select_rows('* from [{}] LIMIT 100'.format(path)))
    if not rows:
        print("No data")
        return

    for row in rows:
        rule = text_format.Parse(row['Rule'], log_whitelist_pb2.TLogWhitelistRule())
        print("Rule: {}, Ticket: {}, Reason: {}. Hash: {}".format(
            format_rule(rule), row['Ticket'], row['Reason'], row['Hash']
        ))


def main():
    parser = argparse.ArgumentParser(prog='Manipulate log whitelist rules')
    parser.add_argument('--create', dest='action', action='store_const', const='create', help="Create rule table")
    parser.add_argument('--add', dest='action', action='store_const', const='add', help="Add rule")
    parser.add_argument('--del', dest='action', action='store_const', const='del', help="Delete rules")
    parser.add_argument('--list', dest='action', action='store_const', const='list', help="List available rules")
    parser.add_argument('--hash', nargs='*', help="Use with --del to delete rules by hashes")
    parser.add_argument('-g', '--genericids', nargs='*',
                        help="Log all messages with GIDS. Use text format TYPE VALUE, i.e: puid 123")
    parser.add_argument('-c', '--cryptaids', nargs='*', help="Log all messages with CRYPTAIDS")
    parser.add_argument('-s', '--shards', nargs='*', help="Log all messages in SHARDS")
    parser.add_argument('--ticket', default='', help="Ticket, related to the rule")
    parser.add_argument('--reason', default='', help="Reason the rule was added")
    parser.add_argument('--path', default=DEFAULT_TABLE, help="Path to whitelist table")

    args = parser.parse_args()

    print("Using table '{}'".format(args.path))

    if not args.action:
        print("Use --add/--del/--list/--create to manipulate rules table")
        parser.print_help()
        return

    client = yt.YtClient(proxy=os.getenv('YT_PROXY', 'hahn'), token=os.getenv('YT_TOKEN'))

    if args.action == 'list':
        list_rows(client, args.path)
        return

    rule = log_whitelist_pb2.TLogWhitelistRule()

    if args.genericids is not None:
        gids = args.genericids
        if (len(gids) % 2):
            print("Can't convert {} to a list of GenericIDs. "
                  "Expecting event number of arguments "
                  "TYPE1 VALUE1 TYPE2 VALUE2 ...".format(gids))
            return
        for i in range(0, len(gids), 2):
            gid = GenericID(gids[i], gids[i+1])
            rule.Vertices.append(gid.to_proto())

    for cid in args.cryptaids or []:
        cid = int(cid)
        rule.Cryptaids.append(cid)
    for shard_id in args.shards or []:
        shard_id = int(shard_id)
        rule.Shards.append(shard_id)

    action = args.action
    if action == 'create':
        print("Createing table {}".format(args.path))
        schema = [
            {'name': 'Hash', 'type': 'uint64', 'sort_order': 'ascending', 'expression': 'farm_hash(Rule)'},
            {'name': 'Rule', 'type': 'string', 'sort_order': 'ascending'},
            {'name': 'Ticket', 'type': 'string'},
            {'name': 'Reason', 'type': 'string'},
        ]
        client.create('table', args.path, recursive=True,
                      attributes={"schema": schema, "dynamic": True})
        client.mount_table(args.path)
    elif action == 'add':
        print('Adding Rule: {}'.format(format_rule(rule)))
        client.insert_rows(args.path, [{
            'Rule': text_format.MessageToString(rule),
            'Ticket': args.ticket,
            'Reason': args.reason,
        }])
    elif action == 'del':
        if args.hash:
            print('Dropping Rules with hashes: {}'.format(args.hash))
            rows = client.select_rows('Rule from [{}] WHERE Hash IN ({})'.format(
                args.path, ','.join(args.hash)))
            drop_data = [{'Rule': row['Rule']} for row in rows]
        else:
            print('Dropping Rule: {}'.format(format_rule(rule)))
            drop_data = [{'Rule': text_format.MessageToString(rule)}]
        client.delete_rows(args.path, drop_data)
    else:
        print("Unknown action {}".format(action))


if __name__ == '__main__':
    main()
