#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/blockcodecs/codecs.h>
#include <library/cpp/getopt/last_getopt.h>

#include <crypta/graph/engine/proto/graph.pb.h>
#include <crypta/graph/engine/graph/lib/split.h>
#include <crypta/graph/rt/sklejka/michurin/proto/state.pb.h>

#include <mapreduce/yt/interface/client.h>

#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/system/env.h>

using TGraph = NCrypta::NGraphEngine::TGraph;
using namespace NLastGetopt;

TString DEFAULT_PATH{"//home/crypta/production/rtsklejka/state/michurin_state"};

void coutComponentSizes(const auto& splitGraph) {
    auto numOfComponens = *MaxElement(splitGraph.begin(), splitGraph.end()) + 1;
    for (ui64 i = 0; i < numOfComponens; ++i) {
        int ans = 0;
        for (auto vertexColor : splitGraph) {
            if (vertexColor == i) {
                ++ans;
            }
        }
        Cout << ans << ' ';
    }
}

int main(int argc, const char* argv[])
{
    NYT::Initialize(argc, argv);

    TOpts opts = TOpts::Default();
    opts.SetFreeArgsMin(1);
    opts.SetFreeArgDefaultTitle("cryptaid", "cryptaids of graphs to split");

    TOpt& mrPathOpt = opts.AddLongOption('p', "path", "Path to table of states.").DefaultValue(DEFAULT_PATH).Optional().OptionalArgument();
    opts.AddLongOption('s', "show-vertices", "use to print vertices").Optional().NoArgument();

    TOptsParseResult res(&opts, argc, argv);
    bool showVertices = res.Has('s');
    auto CIDs = res.GetFreeArgs();

    TString PATH = res.Get<TString>(&mrPathOpt);
    auto client = NYT::CreateClient(GetEnv("YT_PROXY"));

    Cout << "Path: " << PATH << Endl;
    for (const auto& CID : CIDs) {
        Cout
            << "----------------------------------------------------------------------------------------------------------------"
            << Endl;
        ui64 cid;
        try {
            cid = FromString<ui64>(CID);
        } catch (...) {
            Cout << "Bad cryptaid: " << CID << Endl;
            continue;
        }

        auto q = TStringBuilder{} << "* FROM [" << PATH << "] WHERE Id = " << cid;
        auto reader = client->SelectRows(q, {});

        if (reader.size() == 0) {
            Cout << "Error: no data for cryptaId " << CID << Endl;
            continue;
        }

        for (const auto& row : reader) {
            TString codec = row.ChildConvertTo<TString>("Codec");
            TString cid = row.ChildConvertTo<TString>("Id");
            TString graphStringZip = row.ChildConvertTo<TString>("State");

            const auto* codecPtr = NBlockCodecs::Codec(codec);
            auto graphString = codecPtr->Decode(graphStringZip);

            TMichurinState state{};
            Y_PROTOBUF_SUPPRESS_NODISCARD state.ParseFromString(graphString);
            const auto& graph = state.GetGraph();
            auto rowGraph = NCrypta::NGraphEngine::TCommonGraph(graph, true);
            auto innerGraph = rowGraph.GetInnerGraph();

            if (innerGraph.Vertices.size() == 0) {
                Cout << "Error: graph has no vertices" << Endl;
                continue;
            }

            auto splitGraph = NCrypta::NGraphEngine::Split(innerGraph);

            Cout << "crytptaId: " << CID << Endl;
            Cout << "Number of vertices:" << innerGraph.Vertices.size() << Endl;
            Cout << "Number of components is: " << *MaxElement(splitGraph.begin(), splitGraph.end()) + 1 << '\n'
                 << Endl;
            if (showVertices) {
                Cout << "Below the vertices are in the following form" << Endl;
                Cout << "componentNumber -- vertexIndex  idType  idValue" << '\n'
                     << Endl;
                for (const auto& vertex : innerGraph.Vertices) {
                    TString idType = NCrypta::NSoup::IdType(vertex.second.CustomVertex.Type).GetName();
                    Cout << splitGraph[vertex.first] << " -- " << vertex.first << ' ' << idType << ' '
                         << vertex.second.CustomVertex.Value << Endl;
                }

                for (const auto& vertexColor : splitGraph) {
                    Cout << vertexColor << ' ';
                }

                Cout << Endl;
            }
            Cout << "Components sizes: ";
            coutComponentSizes(splitGraph);
            Cout << Endl;
        }
    }
    return 0;
}
