#include <crypta/graph/rt/sklejka/cid_resolver/lib/resolver.h>
#include <crypta/graph/rt/sklejka/resharder/bin/config.pb.h>
#include <crypta/graph/rt/sklejka/rows_processor/proto/config.pb.h>
#include <crypta/graph/rt/sklejka/rows_processor/common.h>
#include <crypta/graph/rt/services/services.h>

#include <build/scripts/c_templates/svnversion.h>
#include <logfeller/lib/parsing/config_storage/config_storage.h>
#include <util/datetime/base.h>

#include <ads/bsyeti/big_rt/lib/utility/profiling/safe_stats_over_yt.h>
#include <ads/bsyeti/big_rt/lib/processing/resharder/lib/resharder.h>
#include <ads/bsyeti/big_rt/lib/processing/resharder/lib/writers.h>
#include <ads/bsyeti/libs/profiling/solomon/exporter.h>
#include <ads/bsyeti/libs/ytex/http/server.h>
#include <ads/bsyeti/libs/ytex/http/std_handlers.h>
#include <ads/bsyeti/libs/ytex/program/program.h>

namespace {

    const TString RTSKLEJKA_RESHARDER = "rtsklejka_resharder";

    // TODO(k-zaitsev): Replace with library version after https://a.yandex-team.ru/review/2191109 is merged
    struct TConditionalHeavyPool {
        explicit TConditionalHeavyPool(ui32 threadsCount) {
            if (threadsCount != 0) {
                HeavyPool = NYT::New<NYT::NConcurrency::TThreadPool>(threadsCount, "resharder_heavy");
                Invoker = NYT::NConcurrency::CreatePrioritizedInvoker(HeavyPool->GetInvoker());
            } else {
                Invoker = NYT::NConcurrency::CreateFakePrioritizedInvoker(NYT::GetSyncInvoker());
            }
        }

        NYT::NConcurrency::TThreadPoolPtr HeavyPool;
        NYT::IPrioritizedInvokerPtr Invoker;
    };

    void StaticSensorsWrite(NSFStats::TSolomonContext ctx) {
        using TLastMetric = NSFStats::TLastMetric<ui64>;
        NSFStats::TSolomonContext sctx{ctx.Detached(), {{"place", "/"}}};
        sctx.Get<TLastMetric>("revision").Set(GetProgramSvnRevision());
        sctx.Get<TLastMetric>("patch").Set(GetArcadiaPatchNumber());
        sctx.Get<TLastMetric>("build_time").Set(GetProgramBuildTimestamp());
        sctx.Get<TLastMetric>("start_time").Set(TInstant::Now().Seconds());
    }

    void RunResharding(const NResharder::TResharderConfig& config,
                       const NYT::TCancelableContextPtr& context) {
        NLogFeller::NParsing::UseEmbeddedConfigs();

        auto solomonExporter = NBSYeti::NProfiling::CreateSolomonExporter(config.GetExporter());
        solomonExporter->Start();

        auto httpServer = NYTEx::NHttp::CreateServer(config.GetHttpServer());
        NYTEx::NHttp::AddStandardHandlers(httpServer, context, config, BIND([solomonExporter] {
                                              return NYT::TSharedRef::FromString(
                                                  NBSYeti::NProfiling::GetSensors(solomonExporter));
                                          }));
        httpServer->Start();

        auto rootCtx = NBigRT::MakeSolomonContext({});
        NBSYeti::TTvmManagerPtr tvmManager{
            config.HasTvmConfig() ? NBSYeti::CreateTvmManager(config.GetTvmConfig()) : NBSYeti::TTvmManagerPtr{}};

        NPersQueue::TPQLibSettings lbPqLibSettings;
        lbPqLibSettings.ThreadsCount = 20;
        lbPqLibSettings.GRpcThreads = 20;
        lbPqLibSettings.ChannelCreationTimeout = TDuration::Seconds(10);
        lbPqLibSettings.DefaultLogger = new TTPQLibGlobalLogBridge();
        NPersQueue::TPQLib lbPqLib(lbPqLibSettings);

        auto ytClients = NYTEx::NRpc::CreateClientsCache();

        auto transactionKeeper = NYTEx::CreateTransactionKeeper(
            TDuration::MilliSeconds(config.GetSharedTransactionPeriod()));

        NBSYeti::TStopToken stopToken(context);

        // svn revision sensor write
        StaticSensorsWrite(rootCtx);

        TConditionalHeavyPool conditionalHeavyPool{config.GetHeavyWorkerThreads()};
        auto writeThrottler{NYT::New<NBigRT::TCommonThrottler>(config.GetYtWriteThrottlerQuota())};

        auto cIdResolver{config.HasCryptaIdResolverConfig() ? CreateCryptaIdResolver(config.GetCryptaIdResolverConfig())
                                                            : nullptr};

        auto destinations = NBigRT::MakeDestinationsMap(config.GetDestinations());
        auto [qytWriters, swiftWriterFactories] = NBigRT::MakeWriters(destinations, ytClients);

        TVector<std::pair<NYT::NConcurrency::TActionQueuePtr, NYT::TFuture<void>>> resharderLauncherThreads;
        for (const auto& eachResharderConfig : config.GetInstances()) {
            if (!eachResharderConfig.GetInstance().GetEnabled()) {
                continue;
            }

            const auto& outputQueue = eachResharderConfig.GetOutputQueue();
            const NBigRT::IRowsProcessorPtr rowsProcessor = NResharder::MakeRowsProcessor(eachResharderConfig.GetInstance().GetServantLabel(),
                                                                                          eachResharderConfig.GetRowsProcessor(),
                                                                                          destinations.at(outputQueue).GetShardsCount(),
                                                                                          outputQueue,
                                                                                          eachResharderConfig.GetSampleShardsMax(),
                                                                                          cIdResolver);

            auto actionQueue{NYT::New<NYT::NConcurrency::TActionQueue>()};
            auto fiberFuture{
                BIND(NBigRT::TResharderLauncher{
                         eachResharderConfig.GetInstance(),
                         config.GetYtCluster(),
                         transactionKeeper,
                         tvmManager,
                         lbPqLib,
                         rootCtx,
                         stopToken,
                         conditionalHeavyPool.Invoker,
                         writeThrottler,
                         rowsProcessor,
                         qytWriters,
                         swiftWriterFactories,
                     })
                    .AsyncVia(actionQueue->GetInvoker())
                    .Run()
                    .ToUncancelable()};
            resharderLauncherThreads.emplace_back(actionQueue, fiberFuture);
        }
        for (auto& [actionQueue, fiberFuture] : resharderLauncherThreads) {
            Y_UNUSED(NYT::NConcurrency::WaitFor(fiberFuture));
            actionQueue->Shutdown();
            actionQueue = {};
        }
    }

    class TShardingProgram: public NYTEx::TProgram<NResharder::TResharderConfig> {
    public:
        TShardingProgram()
            : NYTEx::TProgram<NResharder::TResharderConfig>(RTSKLEJKA_RESHARDER) {
        }

        int DoRun(const NYT::TCancelableContextPtr& context) override {
            RunResharding(Config(), context);
            return 0;
        }
    };
} // namespace

int main(int argc, const char** argv) {
    return NYTEx::RunProgram<TShardingProgram>(argc, argv);
}
