#include "converter.h"

#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/graph/soup/config/cpp/soup_config.h>

namespace NRtSklejka {
    TConverterFromLinks::TConverterFromLinks(bool validateEdge)
        : EdgeTypeWhitelist(CreateWhiteList())
        , ValidateEdge(validateEdge)
    {
    }

    THashSet<TConverterFromLinks::TEdgeTypeTuple> TConverterFromLinks::CreateWhiteList() const {
        THashSet<std::pair<NIdType::EIdType, NIdType::EIdType>> mmWhitelist;
        for (const auto& et : NCrypta::NSoup::EdgeTypes()) {
            if (et.GetLogSource() == NLogSource::METRIKA_MOBILE_LOG) {
                mmWhitelist.emplace(std::make_pair(et.GetId1Type(), et.GetId2Type()));
            }
        }

        THashSet<TConverterFromLinks::TEdgeTypeTuple> result;
        for (const auto& et : NCrypta::NSoup::EdgeTypes()) {
            if (et.GetSourceType() == NCrypta::NSoup::NSourceType::RTDI) {
                const auto t = std::tuple(et.GetId1Type(), et.GetId2Type(), et.GetSourceType(), et.GetLogSource());
                result.emplace(std::move(t));

            } else if (et.GetSourceType() == NCrypta::NSoup::NSourceType::RTMR && mmWhitelist.contains(std::make_pair(et.GetId1Type(), et.GetId2Type()))) {
                const auto t = std::make_tuple(et.GetId1Type(), et.GetId2Type(), et.GetSourceType(), et.GetLogSource());
                result.emplace(std::move(t));
            }
        }

        return result;
    }

    TConverterStats TConverterFromLinks::Convert(
        const NCrypta::NSoup::NBB::TLinks links,
        TVector<NCrypta::NEvent::TSoupEvent>& events) const {
        TVector<NIdentifiers::TGenericID> vertices;
        TConverterStats stats;

        for (const auto& v : links.GetVertices()) {
            NIdentifiers::TGenericID gid(v.GetIdType(), v.GetId());
            if (gid.IsSignificant()) {
                vertices.emplace_back(std::move(gid));
            } else {
                stats.insignificantVertices++;
            }
        }

        for (auto i = vertices.begin(); i != vertices.end(); ++i) {
            for (auto j = std::next(i); j != vertices.end(); ++j) {
                const auto& v1 = *i;
                const auto& v2 = *j;

                NCrypta::NSoup::TEdgeType edgeType;
                try {
                    edgeType = NCrypta::NSoup::EdgeType(
                        v1.GetType(),
                        v2.GetType(),
                        links.GetSourceType(),
                        links.GetLogSource());
                } catch (...) {
                    continue;
                }

                const auto usage = NCrypta::NSoup::EdgeUsage(edgeType);
                if (ValidateEdge && !usage.GetRtSklejka()) {
                    stats.nonRtSkejkaEdges++;
                    continue;
                }

                if (!EdgeTypeWhitelist.contains(
                        std::make_tuple(edgeType.GetId1Type(),
                                        edgeType.GetId2Type(),
                                        edgeType.GetSourceType(),
                                        edgeType.GetLogSource()))) {
                    continue;
                }

                NCrypta::NEvent::TSoupEvent event;
                NCrypta::NGraphEngine::TEdgeBetween edge;
                edge.SetLogSource(links.GetLogSource());
                edge.SetSourceType(links.GetSourceType());

                // NOTE(k-zaitsev) EdgeType checks both directions v1-v2 and v2-v1 and returns correct orientation
                const auto& [first, second] = v1.GetType() == edgeType.GetId1Type() ? std::make_tuple(v1, v2) : std::make_tuple(v2, v1);
                edge.MutableVertex1()->CopyFrom(first.ToProto());
                edge.MutableVertex2()->CopyFrom(second.ToProto());

                event.MutableEdge()->Swap(&edge);
                event.SetUnixtime(links.GetLogEventTimestamp());
                event.SetSource(NCrypta::NEvent::ESource::LINKS);

                events.emplace_back(std::move(event));
            }
        }

        return stats;
    }
}
