#include "common.h"
#include "parsers.h"
#include "row.h"

#include <crypta/graph/engine/proto/graph.pb.h>
#include <crypta/graph/soup/config/cpp/soup_config.h>
#include <crypta/lib/native/proto_serializer/proto_serializer.h>

#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/yson/node/node_io.h>
#include <util/string/strip.h>

namespace {
    const TString INSIGNIFICANT_VERTICES_KEY("skipped_insignificant_vertices");
    const TString NON_RTSKEJKA_EDGES_KEY("skipped_non_rtsklejka_edges");
}

namespace NResharder {
    using TParserConfig = TRowsProcessorConfig::TParser::TFormat;
    using NCrypta::NIdentifiersProto::NIdType::EIdType_Name;
    using NCrypta::NSoup::NLogSource::ELogSourceType_Name;
    using NCrypta::NSoup::NSourceType::ESourceType_Name;
    using namespace NResharder::NExceptions;

    // soup
    TParserSoup::TParserSoup(const TParserConfig& config)
        : Isb64Encoded(config.GetB64Encoding())
        , ValidateEdge(config.GetValidateEdge())
        , MaxRewindCounter(config.GetMaxRewindCounter())
    {
    }

    ISoupParser::TParseResult TParserSoup::Parse(TStringBuf text, NSFStats::TSolomonContext& ctx) const {
        NCrypta::NEvent::TSoupEvent msg;

        Y_PROTOBUF_SUPPRESS_NODISCARD msg.ParseFromString(
            Isb64Encoded ? Base64Decode(Strip(TString{text.data(), text.size()}))
                         : TString{text.data(), text.size()});

        if (msg.GetCounter() >= MaxRewindCounter) {
            ythrow TTooMuchRewinds() << "Rewind loopback limit: " << msg.GetCounter() << " >= " << MaxRewindCounter;
        }

        if (ValidateEdge) {
            const auto stats = Validate(msg);

            if (stats.insignificantVertices || stats.nonRtSkejkaEdges) {
                ctx.Get<NSFStats::TSumMetric<ui64>>(INSIGNIFICANT_VERTICES_KEY).Inc(stats.insignificantVertices);
                ctx.Get<NSFStats::TSumMetric<ui64>>(NON_RTSKEJKA_EDGES_KEY).Inc(stats.nonRtSkejkaEdges);
                ythrow TNoAppropriateEdges() << "Skipped non appropriate edge from soup: " << NCrypta::NProtoSerializer::ToString(msg);
            }
        }

        return {std::move(msg)};
    }

    NRtSklejka::TConverterStats TParserSoup::Validate(const NCrypta::NEvent::TSoupEvent& msg) const {
        uint32_t insignificantVertices = 0, nonRtSkejkaEdges = 0;

        for (const auto& gid : {msg.GetEdge().GetVertex1(), msg.GetEdge().GetVertex2()}) {
            if (!NIdentifiers::TGenericID(gid).IsSignificant()) {
                insignificantVertices++;
            }
        }

        const auto edge{
            NCrypta::NSoup::EdgeType(
                msg.GetEdge().GetVertex1().GetType(),
                msg.GetEdge().GetVertex2().GetType(),
                msg.GetEdge().GetSourceType(),
                msg.GetEdge().GetLogSource())};

        if (!NCrypta::NSoup::EdgeUsage(edge).GetRtSklejka()) {
            nonRtSkejkaEdges++;
        }

        return {
            .nonRtSkejkaEdges = nonRtSkejkaEdges,
            .insignificantVertices = insignificantVertices,
        };
    }

    // links
    TParserLinks::TParserLinks(const TParserConfig& config)
        : Isb64Encoded(config.GetB64Encoding())
        , Debounce("deb", config.GetDebounceCfg())
        , MaxLinksAge(config.GetMaxLinksAge())
        , Converter(config.GetValidateEdge())
    {
    }

    ISoupParser::TParseResult TParserLinks::Parse(TStringBuf text, NSFStats::TSolomonContext& ctx) const {
        NCrypta::NSoup::NBB::TLinks links;

        Y_PROTOBUF_SUPPRESS_NODISCARD links.ParseFromString(
            Isb64Encoded ? Base64Decode(Strip(TString{text.data(), text.size()}))
                         : TString{text.data(), text.size()});

        Validate(links);

        ISoupParser::TParseResult events;
        const auto stats = Converter.Convert(links, events);

        events.erase(std::remove_if(events.begin(), events.end(), [&](const auto& event) { return IsInDebounceCache(event); }), events.end());

        ctx.Get<NSFStats::TSumMetric<ui64>>(INSIGNIFICANT_VERTICES_KEY).Inc(stats.insignificantVertices);
        ctx.Get<NSFStats::TSumMetric<ui64>>(NON_RTSKEJKA_EDGES_KEY).Inc(stats.nonRtSkejkaEdges);

        if (events.empty()) {
            ythrow TNoAppropriateEdges() << "TLinks w/ no appropriate edges: " << NCrypta::NProtoSerializer::ToString(links);
        }

        return events;
    }

    bool TParserLinks::IsInDebounceCache(const NCrypta::NEvent::TSoupEvent& event) const {
        TString serialized;
        Y_PROTOBUF_SUPPRESS_NODISCARD event.GetEdge().SerializeToString(&serialized);

        if (Debounce.Pull(serialized)) {
            return true;
        }

        Debounce.Push(serialized);
        return false;
    }

    void TParserLinks::Validate(const NCrypta::NSoup::NBB::TLinks& links) const {
        const static int minVerticesRequired{2};
        if (links.GetVertices().size() < minVerticesRequired) {
            ythrow TTooLittleVertices() << "Skipping event on vertices size: " << links.GetVertices().size() << " < " << minVerticesRequired;
        }

        const auto& eventTimestamp = links.GetLogEventTimestamp();
        const auto& now = TInstant::Now().Seconds();
        if (now - eventTimestamp > MaxLinksAge) {
            ythrow TTooOldLinks() << "Skipping event on timestamp: " << eventTimestamp << ", while now is:" << now;
        }
    }

    // bookeeping
    TParserBookkeeping::TParserBookkeeping(const TParserConfig& config)
        : EventType(config.GetEventType())
    {
        if (EventType == NCrypta::NEvent::TMichurinBookkeepingEvent::UNDEFINED) {
            ythrow yexception() << "EventType is undefined.";
        }
    }

    IBookParser::TParseResult TParserBookkeeping::Parse(TStringBuf text, NSFStats::TSolomonContext&) const {
        auto textNodeMap = NYT::NodeFromYsonString(text).AsMap();

        IBookParser::TParseResult msg;
        msg.SetType(EventType);
        msg.SetCryptaId(textNodeMap["Id"].AsUint64());

        return msg;
    }

    // empty
    TParserEmpty::TParserEmpty(const TParserConfig& /* config */) {
    }

    ISoupParser::TParseResult TParserEmpty::Parse(TStringBuf, NSFStats::TSolomonContext&) const {
        ythrow yexception() << "TParserEmpty::Parse ythrow";
        return {};
    }

    // parser makers
    template <>
    NRtSklejka::IParserPtr<ISoupParser::TParseResult> MakeParser(const TRowsProcessorConfig::TParser& config) {
        if (config.HasFromSoup()) {
            return MakeHolder<TParserSoup>(config.GetFromSoup());
        } else if (config.HasFromLinks()) {
            return MakeHolder<TParserLinks>(config.GetFromLinks());
        } else if (config.HasFromEmpty()) {
            return MakeHolder<TParserEmpty>(config.GetFromEmpty());
        } else {
            ythrow yexception() << "there is no row format in config!";
        }
    }

    template <>
    NRtSklejka::IParserPtr<IBookParser::TParseResult> MakeParser(const TRowsProcessorConfig::TParser& config) {
        if (config.HasFromBookkeeping()) {
            return MakeHolder<TParserBookkeeping>(config.GetFromBookkeeping());
        } else {
            ythrow yexception() << "there is no row format in config!";
        }
    }
}
