#include "common.h"
#include "rows_processor.h"

#include <crypta/graph/rt/events/events.h>
#include <crypta/graph/rt/events/proto/soup.pb.h>

#include <logfeller/lib/chunk_splitter/chunk_splitter.h>

#include <library/cpp/logger/global/global.h>

namespace NResharder {
    using namespace NResharder::NExceptions;

    TRowsProcessor::TRowsProcessor(const TRowsProcessorConfig& config,
                                   const ui32 reshardingModule,
                                   const TString& destinationQueue,
                                   const ui64 sampleShardsMax,
                                   const NCrypta::TCryptaIdResolverPtr cIdResolver)
        : TBaseRowsProcessor(config, reshardingModule, destinationQueue, sampleShardsMax)
        , Parser(MakeParser<ISoupParser::TParseResult>(config.GetParser()))
        , CIdResolver(cIdResolver)
    {
    }

    void TRowsProcessor::Process(NSFStats::TSolomonContext& ctx, TInstant now, NBigRT::TRowsBatch& rows) const {
        if (CIdResolver != nullptr) {
            ResolveCryptaIds(ctx, now, rows);
        }

        for (auto& row : rows) {
            if (row.MessageType == NCrypta::NEvent::EMessageType::SOUP) {
                auto msg{dynamic_cast<NCrypta::NEvent::TSoupEvent*>(row.Message.Get())};
                auto fields{dynamic_cast<TRowFields*>(row.Fields.Get())};

                msg->SetCryptaId1(fields->CryptaIds.size() > 0 ? fields->CryptaIds[0] : 0);
                msg->SetCryptaId2(fields->CryptaIds.size() > 1 ? fields->CryptaIds[1] : 0);
            }
        }
    }

    void TRowsProcessor::ParseRecord(const TStringBuf& record, NSFStats::TSolomonContext& ctx, NBigRT::TRowsBatch& rows,
                                     const NBigRT::TRowMeta& meta) const {
        for (auto& event : Parser->Parse(record, ctx)) {
            NBigRT::TRow row{
                .Meta = meta,
                .TimeStamp = TInstant::Seconds(event.GetUnixtime()),
                .MessageType = static_cast<ui64>(MessageType),
                .Message = NCrypta::NEvent::MakeMessage(MessageType),
                .Fields = MakeHolder<TRowFields>(),
            };

            const auto fields = static_cast<TRowFields*>(row.Fields.Get());
            fields->CryptaIds = {0, 0};
            fields->Ids.emplace_back(std::move(NIdentifiers::TGenericID(event.GetEdge().GetVertex1())));
            fields->Ids.emplace_back(std::move(NIdentifiers::TGenericID(event.GetEdge().GetVertex2())));

            const auto msg{static_cast<NCrypta::NEvent::TSoupEvent*>(row.Message.Get())};
            msg->Swap(&event);
            Y_ENSURE(row.Message->IsInitialized(), "Not all of required fields are initialized!");

            rows.push_back(std::move(row));
        }
    }

    void TRowsProcessor::Parse(NSFStats::TSolomonContext& ctx, const NBigRT::TRowMeta& meta, TStringBuf message,
                               NBigRT::TRowsBatch& rows) const {
        NLogFeller::NChunkSplitter::TRecordContext context;
        TStringBuf record, skip;
        auto splitter{NLogFeller::NChunkSplitter::CreateChunkSplitter(Splitter)};

        for (auto iterator = splitter->CreateIterator(message); iterator.NextRecord(record, skip, context);) {
            ctx.Get<NSFStats::TSumMetric<ui64>>("parse_rows").Inc(1);
            try {
                ParseRecord(record, ctx, rows, meta);
            } catch (const TInvalidMessageError&) {
                ctx.Get<NSFStats::TSumMetric<ui64>>("rows_count_skipped_by_invalid_message").Inc(1);
            } catch (const TTooLittleVertices&) {
                ctx.Get<NSFStats::TSumMetric<ui64>>("rows_count_skipped_by_too_little_vertices").Inc(1);
            } catch (const TTooOldLinks&) {
                ctx.Get<NSFStats::TSumMetric<ui64>>("rows_count_skipped_by_too_old_links").Inc(1);
            } catch (const TNoAppropriateEdges& err) {
                ctx.Get<NSFStats::TSumMetric<ui64>>("rows_count_skipped_by_no_appropriate_edgees").Inc(1);
            } catch (const TTooMuchRewinds& err) {
                ctx.Get<NSFStats::TSumMetric<ui64>>("rows_skipped_by_too_many_rewinds").Inc(1);
            } catch (...) {
                ctx.Get<NSFStats::TSumMetric<ui64>>("rows_count_skipped_by_error").Inc(1);
                DEBUG_LOG << "Skipped: type=" << static_cast<i64>(MessageType)
                          << ", record='" << record << "', error=" << CurrentExceptionMessage() << '\n';
            }
        }

        ctx.Get<NSFStats::TSumMetric<ui64>>("parse_messages").Inc(1);
    }

    void TRowsProcessor::ResolveCryptaIds(NSFStats::TSolomonContext& ctx, TInstant now, NBigRT::TRowsBatch& rows) const {
        Y_UNUSED(now);

        NSFStats::TSolomonContext identificationCtx(ctx, {{"place", "identification"}});
        auto resolveCryptaIdCnt{identificationCtx.Get<NSFStats::TSumMetric<ui64>>("id_resolve_crypta_id")};
        auto presetCryptaIdCnt{identificationCtx.Get<NSFStats::TSumMetric<ui64>>("id_preset_crypta_id")};
        auto missingCryptaIdCnt{identificationCtx.Get<NSFStats::TSumMetric<ui64>>("id_missing_crypta_id")};

        TVector<NIdentifiers::TGenericID> requestIds;
        requestIds.reserve(rows.size() * 2);

        // select ids w/o cryptaid to resolve
        for (const auto& row : rows) {
            auto fields{dynamic_cast<TRowFields*>(row.Fields.Get())};
            const auto& msg{dynamic_cast<NCrypta::NEvent::TSoupEvent*>(row.Message.Get())};

            if (!msg->GetCryptaId1()) {
                requestIds.emplace_back(fields->Ids[0]);
            }
            if (!msg->GetCryptaId2()) {
                requestIds.emplace_back(fields->Ids[1]);
            }
        }

        {
            const auto result{CIdResolver->Identify(requestIds, identificationCtx)};
            // TODO maybe change interface from vector to hashmap id -> cryptaId
            // fill resolved cryptaIds, this cycle should be the very same as the one above
            size_t index{0};
            for (auto& row : rows) {
                if (row.MessageType != NCrypta::NEvent::EMessageType::SOUP) {
                    ERROR_LOG << "Skipping  Non-SOUP event: " << NCrypta::NEvent::EMessageType_Name(row.MessageType);
                    continue;
                }
                auto fields{dynamic_cast<TRowFields*>(row.Fields.Get())};
                const auto& msg{dynamic_cast<NCrypta::NEvent::TSoupEvent*>(row.Message.Get())};

                auto SetOrResolveCryptaId = [&](const auto& presetCid) {
                    if (presetCid) {
                        presetCryptaIdCnt.Inc(1);
                        return presetCid;
                    }

                    ui64 resolvedCid{0};
                    if (result[index]) {
                        resolvedCid = *result[index];
                        resolveCryptaIdCnt.Inc(1);
                    } else {
                        missingCryptaIdCnt.Inc(1);
                    }
                    ++index;
                    return resolvedCid;
                };
                fields->CryptaIds[0] = SetOrResolveCryptaId(msg->GetCryptaId1());
                fields->CryptaIds[1] = SetOrResolveCryptaId(msg->GetCryptaId2());
                if (msg->GetReversed()) {
                    fields->CryptaId = fields->CryptaIds[1] ? fields->CryptaIds[1] : fields->CryptaIds[0];
                } else {
                    fields->CryptaId = fields->CryptaIds[0] ? fields->CryptaIds[0] : fields->CryptaIds[1];
                }
            }
        }
    }
}
