#include "common.h"
#include "rows_processor_bookkeeping.h"

#include <crypta/graph/rt/events/events.h>

#include <logfeller/lib/chunk_splitter/chunk_splitter.h>

#include <library/cpp/logger/global/global.h>

namespace NResharder {
    TBookkeepingRowsProcessor::TBookkeepingRowsProcessor(const TRowsProcessorConfig& config,
                                                         const ui32 reshardingModule,
                                                         const TString& destinationQueue,
                                                         const ui64 sampleShardsMax)
        : TBaseRowsProcessor(config, reshardingModule, destinationQueue, sampleShardsMax)
        , Parser(MakeParser<IBookParser::TParseResult>(config.GetParser()))
    {
    }

    // TODO(@zheglov) this function is very similiar to the one in rows_processor.cpp
    // should probably make it templated by Parser::TParseResult
    void TBookkeepingRowsProcessor::ParseRecord(const TStringBuf& record, NSFStats::TSolomonContext& ctx,
                                                NBigRT::TRowsBatch& rows, const NBigRT::TRowMeta& meta) const {
        NBigRT::TRow row{
            .Meta = meta,
            .TimeStamp = TInstant::Now(),
            .MessageType = static_cast<ui64>(MessageType),
            .Message = NCrypta::NEvent::MakeMessage(MessageType),
            .Fields = MakeHolder<TRowFields>(),
        };

        auto msg = Parser->Parse(record, ctx);
        Y_ENSURE(msg.IsInitialized(), "Not all of required fields are initialized!");

        const auto fields = static_cast<TRowFields*>(row.Fields.Get());
        fields->CryptaId = msg.GetCryptaId();

        const auto rowMsg{static_cast<NCrypta::NEvent::TMichurinBookkeepingEvent*>(row.Message.Get())};
        rowMsg->Swap(&msg);

        rows.push_back(std::move(row));
    }

    void TBookkeepingRowsProcessor::Parse(NSFStats::TSolomonContext& ctx, const NBigRT::TRowMeta& meta, TStringBuf message,
                                          NBigRT::TRowsBatch& rows) const {
        TStringBuf record, skip;
        NLogFeller::NChunkSplitter::TRecordContext context;
        auto splitter{NLogFeller::NChunkSplitter::CreateChunkSplitter(Splitter)};

        for (auto iterator = splitter->CreateIterator(message); iterator.NextRecord(record, skip, context);) {
            ctx.Get<NSFStats::TSumMetric<ui64>>("parse_rows").Inc(1);
            try {
                ParseRecord(record, ctx, rows, meta);
            } catch (...) {
                ERROR_LOG << "Skipped: type=" << static_cast<i64>(MessageType)
                          << ", record='" << record << "', error=" << CurrentExceptionMessage() << '\n';
                ctx.Get<NSFStats::TSumMetric<ui64>>("rows_count_skipped_by_error").Inc(1);
            }
        }
        ctx.Get<NSFStats::TSumMetric<ui64>>("parse_messages").Inc(1);
    }
}
