#include "common.h"

#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/graph/soup/config/proto/bigb.pb.h>
#include <crypta/graph/soup/config/cpp/soup_config.h>

#include <crypta/graph/rt/sklejka/cid_resolver/lib/resolver.h>
#include <crypta/graph/rt/events/events.h>
#include <crypta/graph/rt/events/proto/event.pb.h>
#include <crypta/graph/rt/events/proto/soup.pb.h>
#include <crypta/graph/rt/sklejka/rows_processor/proto/config.pb.h>

#include <google/protobuf/util/message_differencer.h>
#include <library/cpp/framing/unpacker.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/vector.h>

using namespace NResharder;
using namespace NIdentifiers;
using google::protobuf::util::MessageDifferencer;

namespace NCrypta {
    class TCryptaIdResolverMock: public ICryptaIdResolver {
    public:
        explicit TCryptaIdResolverMock() {
        }

        TChunkResponse<TCryptaId> Identify(const TChunkRequest& ids, NSFStats::TSolomonContext& ctx) override {
            Y_UNUSED(ctx);

            TChunkResponse<TCryptaId> result;
            result.reserve(ids.size());
            for (size_t index{0}; index < ids.size(); ++index) {
                if (const auto iterator{CidMap.find(ids[index])}; iterator != CidMap.end()) {
                    result.emplace_back(iterator->second);
                } else {
                    ythrow yexception() << "Incorect Id: "
                                        << ids[index].GetTypeString() << " " << ids[index].GetValue() << "\n";
                }
            }
            return result;
        }

    private:
        const THashMap<TGenericID, TMaybe<ui64>> CidMap{
            []() {
                THashMap<TGenericID, TMaybe<ui64>> map;

                map[TGenericID{"email", "some@e.co"}] = 732;
                map[TGenericID{"email_md5", "070b657c757375ca8ce580f7e5ab158b"}] = 601;
                map[TGenericID{"email_sha256", "f617c78ee13f2d74764adf17dabafde1e84b78e6a7b5a37db059f1757b4efc22"}] = 600;
                map[TGenericID{"gaid", "0000"}] = 713;
                map[TGenericID{"gaid", "00023a85-d82e-45b3-b49a-55eaadce8eee"}] = Nothing();
                map[TGenericID{"gaid", "00023a85-d82e-45b3-b49a-55eaadce8fef"}] = 200;
                map[TGenericID{"gaid", "a1"}] = 201;
                map[TGenericID{"gaid", "a3"}] = 202;
                map[TGenericID{"gaid", "a5"}] = 203;
                map[TGenericID{"gaid", "a7"}] = 204;
                map[TGenericID{"gaid", "a9"}] = 205;
                map[TGenericID{"gaid", "abcd"}] = 999;
                map[TGenericID{"idfa", "000045BA-A3D9-4E88-885F-BBD932F29BE3"}] = Nothing();
                map[TGenericID{"idfa", "10"}] = 302;
                map[TGenericID{"idfa", "1234"}] = 999;
                map[TGenericID{"idfa", "a2"}] = 303;
                map[TGenericID{"idfa", "a4"}] = 304;
                map[TGenericID{"idfa", "a6"}] = 305;
                map[TGenericID{"idfa", "a8"}] = 306;
                map[TGenericID{"mm_device_id", "a49d399ef3334bd4811bbaad0474f021"}] = 500;
                map[TGenericID{"phone_md5", "070b657c757372318ce580f7e5ab158b"}] = 501;
                map[TGenericID{"yandexuid", "1234567810"}] = 101;
                map[TGenericID{"yandexuid", "1234567890"}] = 100;
                map[TGenericID{"yandexuid", "1236543210"}] = 999;
                map[TGenericID{"yandexuid", "1236543211"}] = 109;
                map[TGenericID{"yandexuid", "1236543220"}] = 102;
                map[TGenericID{"yandexuid", "1236543230"}] = Nothing();
                map[TGenericID{"yandexuid", "1236543240"}] = 104;
                map[TGenericID{"yandexuid", "1236543250"}] = 105;
                map[TGenericID{"yandexuid", "1236543260"}] = 106;
                map[TGenericID{"yandexuid", "1236543270"}] = 107;
                map[TGenericID{"yandexuid", "1236543280"}] = 108;

                return map;
            }()};
    };
}

TString RowToProtoSoup(const TString& id1, const TString& id1Type,
                       const TString& id2, const TString& id2Type,
                       const ui64 timestamp,
                       const NCrypta::NSoup::NSourceType::ESourceType st = NCrypta::NSoup::NSourceType::APP_METRICA,
                       const NCrypta::NSoup::NLogSource::ELogSourceType ls = NCrypta::NSoup::NLogSource::METRIKA_MOBILE_LOG) {
    NCrypta::NEvent::TSoupEvent message;
    message.SetUnixtime(timestamp);
    message.MutableEdge()->MutableVertex1()->CopyFrom(TGenericID(id1Type, id1).ToProto());
    message.MutableEdge()->MutableVertex2()->CopyFrom(TGenericID(id2Type, id2).ToProto());
    message.MutableEdge()->SetLogSource(ls);
    message.MutableEdge()->SetSourceType(st);
    message.SetCounter(3);
    TString output;
    Y_PROTOBUF_SUPPRESS_NODISCARD message.SerializeToString(&output);
    return Base64Encode(output);
}

Y_UNIT_TEST_SUITE(RowsProcessorTestSuite) {
    Y_UNIT_TEST(TestParseUnknown) {
        const TString configJson{
            R"({
                "MessageType": "SOME_UNKNOWN_MESSAGE",
                "Parser": {
                    "FromEmpty": {
                    }
                }
            })"};

        UNIT_ASSERT_EXCEPTION(MakeRowsProcessor("resharder",
                                                NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson),
                                                3, "someQueue", UINT64_MAX, MakeIntrusive<NCrypta::TCryptaIdResolverMock>()), yexception);
    }

    Y_UNIT_TEST(TestProcessEmpty) {
        const TString configJson{
            R"({
                "ChoiceType": "1",
                "MessageType": "SOUP",
                "Parser": {
                    "FromEmpty": {}
                },
                "ReshardingModule": 3,
                "ShardingAlgorithm": "RoundRobin"
            })"};

        const TString inputRows{
            TStringBuilder{}
            << RowToProtoSoup("abcd", "gaid", "1234567890", "yandexuid", 1567890120ull) << "\n"
            << RowToProtoSoup("1234", "idfa", "1234567890", "yandexuid", 1567890121ull) << "\n"
            << RowToProtoSoup("0000", "gaid", "1236543210", "yandexuid", 1567890122ull)};

        auto rowsProcessor{MakeRowsProcessor("resharder",
                                             NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson),
                                             3, "someQueue", UINT64_MAX, MakeIntrusive<NCrypta::TCryptaIdResolverMock>())};

        NSFStats::TStats stats{};
        NSFStats::TSolomonContext ctx{stats, {}};

        NBigRT::TRowsBatch rows;
        NBigRT::TRowMeta meta;
        rowsProcessor->Parse(ctx, meta, inputRows, rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 0);
        rowsProcessor->Process(ctx, TInstant::Now(), rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 0);
    }

    Y_UNIT_TEST(TestProcessInvalid) {
        const TString configJson{
            R"({
                "ChoiceType": "1",
                "MessageType": "SOUP",
                "Parser": {
                    "FromSoup": {
                        "B64Encoding": true,
                        "ValidateEdge": true,
                        "TimeStamp": "Unixtime"
                    }
                },
                "ReshardingModule": 3,
                "ShardingAlgorithm": "RoundRobin"
            })"};

        const TString inputRows{
            TStringBuilder{}
            // Valid
            << RowToProtoSoup("00023a85-d82e-45b3-b49a-55eaadce8fef", "gaid", "a49d399ef3334bd4811bbaad0474f021", "mm_device_id", 1567890120ull) << "\n"
            // No such edge
            << RowToProtoSoup("00023a85-d82e-45b3-b49a-55eaadce8fef", "gaid", "123", "yandexuid", 1567890121ull) << "\n"
            // Invalid idfa
            << RowToProtoSoup("invalid", "idfa", "123", "yandexuid", 1567890122ull) << "\n"
            // Insignificant gaid
            << RowToProtoSoup("00000000-0000-0000-0000-000000000000", "gaid", "123", "yandexuid", 1567890123ull) << "\n"
            // Valid gaid, insignificant yandexuid
            << RowToProtoSoup("00023a85-d82e-45b3-b49a-55eaadce8fef", "gaid", "0", "yandexuid", 1567890124ull)};

        auto rowsProcessor{MakeRowsProcessor("resharder",
                                             NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson),
                                             3, "someQueue", UINT64_MAX, MakeIntrusive<NCrypta::TCryptaIdResolverMock>())};

        NSFStats::TStats stats{};
        NSFStats::TSolomonContext ctx{stats, {}};

        NBigRT::TRowsBatch rows;
        NBigRT::TRowMeta meta;
        rowsProcessor->Parse(ctx, meta, inputRows, rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 1);
        rowsProcessor->Process(ctx, TInstant::Now(), rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 1);

        {
            auto fields{*dynamic_cast<TRowFields*>(rows[0].Fields.Get())};
            UNIT_ASSERT_VALUES_EQUAL(fields.CryptaId, 200);
        }

        UNIT_ASSERT_VALUES_EQUAL(rows[0].TimeStamp, TInstant::Seconds(1567890120ull));

        THashMap<TString, TVector<NBigRT::TAggregatedRow>> aggregatedRows;
        rowsProcessor->Aggregate(ctx, rows, aggregatedRows);
        UNIT_ASSERT_VALUES_EQUAL(aggregatedRows.size(), 1);
        for (const auto& [queue, aggregatedRowsForQueue] : aggregatedRows) {
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue[0].Shard, 1);
        }
    }

    Y_UNIT_TEST(TestParseTLinks) {
        const TString configJson{
            R"({
                "Splitter": {
                    "LogfellerSplitter": "none"
                },
                "ChoiceType": "1",
                "MessageType": "SOUP",
                "Parser": {
                    "FromLinks": {
                        "B64Encoding": false,
                        "ValidateEdge": true
                    }
                },
                "ReshardingModule": 3,
                "ShardingAlgorithm": "RoundRobin"
            })"};

        const TString linksStr{
            R"({
                "Vertices":[
                    {"IdType":1,"Id":"54498171576176851"},
                    {"IdType":22,"Id":"7aa97a80cf0149d9bf854bf6303ec45a"},
                    {"IdType":24,"Id":"390B2D8F-261F-44A2-8E09-9BF5C58E9B00"},
                    {"IdType":28,"Id":"819134FA-AA77-442C-BF51-61B791490D29"},
                    {"IdType":44,"Id":"390B2D8F-261F-44A2-8E09-9BF5C58E9B01"}
                ],
                "SourceType":95,
                "LogSource":45,
                "Indevice":true,
                "LogEventTimestamp":1607449650,
                "Usage":[1,2]}
            )"};

        NCrypta::NSoup::NBB::TLinks links;
        NProtobufJson::Json2Proto(linksStr, links);
        links.SetLogEventTimestamp(TInstant::Now().Seconds());

        TString input;
        Y_PROTOBUF_SUPPRESS_NODISCARD links.SerializeToString(&input);

        auto rowsProcessor{MakeRowsProcessor("resharder",
                                             NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson),
                                             3, "someQueue", UINT64_MAX, MakeIntrusive<NCrypta::TCryptaIdResolverMock>())};

        NSFStats::TStats stats{};
        NSFStats::TSolomonContext ctx{stats, {}};

        NBigRT::TRowsBatch rows;
        NBigRT::TRowMeta meta;
        rowsProcessor->Parse(ctx, meta, input, rows);

        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 4);

        TVector<std::pair<TString, TString>> result{
            {"54498171576176851", "7aa97a80cf0149d9bf854bf6303ec45a"},
            {"54498171576176851", "390b2d8f261f44a28e099bf5c58e9b00"},
            {"390b2d8f261f44a28e099bf5c58e9b00", "7aa97a80cf0149d9bf854bf6303ec45a"},
            {"819134FA-AA77-442C-BF51-61B791490D29", "390b2d8f261f44a28e099bf5c58e9b00"},
        };
        for (size_t i=0; i<rows.size(); ++i) {
            const auto& msg = static_cast<NCrypta::NEvent::TSoupEvent*>(rows[i].Message.Get());
            UNIT_ASSERT_VALUES_EQUAL(TGenericID(msg->GetEdge().GetVertex1()).GetValue(), result[i].first);
            UNIT_ASSERT_VALUES_EQUAL(TGenericID(msg->GetEdge().GetVertex2()).GetValue(), result[i].second);
        }
    }

    Y_UNIT_TEST(TestProcess) {
        const TString configJson{
            R"({
                "ChoiceType": "1",
                "MessageType": "SOUP",
                "Parser": {
                    "FromSoup": {
                        "B64Encoding": true,
                        "TimeStamp": "Unixtime"
                    }
                },
                "ReshardingModule": 3,
                "ShardingAlgorithm": "RoundRobin"
            })"};

        const TString inputRows{
            TStringBuilder{}
            << RowToProtoSoup("abcd", "gaid", "1234567890", "yandexuid", 1567890120ull) << "\n"
            << RowToProtoSoup("1234", "idfa", "1234567890", "yandexuid", 1567890121ull) << "\n"
            << RowToProtoSoup("0000", "gaid", "1236543210", "yandexuid", 1567890122ull)};

        auto rowsProcessor{MakeRowsProcessor("resharder",
                                             NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson),
                                             3, "someQueue", UINT64_MAX, MakeIntrusive<NCrypta::TCryptaIdResolverMock>())};

        NSFStats::TStats stats{};
        NSFStats::TSolomonContext ctx{stats, {}};

        NBigRT::TRowsBatch rows;
        NBigRT::TRowMeta meta;
        rowsProcessor->Parse(ctx, meta, inputRows, rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 3);
        rowsProcessor->Process(ctx, TInstant::Now(), rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 3);

        {
            auto fields{*dynamic_cast<TRowFields*>(rows[0].Fields.Get())};
            UNIT_ASSERT_VALUES_EQUAL(fields.CryptaId, 999);
        }
        {
            auto fields{*dynamic_cast<TRowFields*>(rows[1].Fields.Get())};
            UNIT_ASSERT_VALUES_EQUAL(fields.CryptaId, 999);
        }
        {
            auto fields{*dynamic_cast<TRowFields*>(rows[2].Fields.Get())};
            UNIT_ASSERT_VALUES_EQUAL(fields.CryptaId, 713);
        }

        UNIT_ASSERT_VALUES_EQUAL(rows[0].TimeStamp, TInstant::Seconds(1567890120ull));
        UNIT_ASSERT_VALUES_EQUAL(rows[1].TimeStamp, TInstant::Seconds(1567890121ull));
        UNIT_ASSERT_VALUES_EQUAL(rows[2].TimeStamp, TInstant::Seconds(1567890122ull));

        THashMap<TString, TVector<NBigRT::TAggregatedRow>> aggregatedRows;
        rowsProcessor->Aggregate(ctx, rows, aggregatedRows);
        UNIT_ASSERT_VALUES_EQUAL(aggregatedRows.size(), 1);
        for (const auto& [queue, aggregatedRowsForQueue] : aggregatedRows) {
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue[0].Shard, 1);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue[1].Shard, 2);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue[2].Shard, 0);
        }
    }

    Y_UNIT_TEST(TestProcessMessage) {
        const TString configJson{
            R"({
                "ChoiceType": "1",
                "MessageType": "SOUP",
                "Parser": {
                    "FromSoup": {
                        "B64Encoding": true,
                        "TimeStamp": "Unixtime"
                    }
                },
                "ReshardingModule": 3,
                "ShardingAlgorithm": "RoundRobin"
            })"};

        const TString inputRows{
            TStringBuilder{}
            << RowToProtoSoup("abcd", "gaid", "1234567890", "yandexuid", 1567890120ull) << "\n"
            << RowToProtoSoup("1234", "idfa", "1234567890", "yandexuid", 1567890121ull) << "\n"
            << RowToProtoSoup("0000", "gaid", "1236543210", "yandexuid", 1567890122ull)};

        auto rowsProcessor{MakeRowsProcessor("resharder",
                                             NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson),
                                             3, "someQueue", UINT64_MAX, MakeIntrusive<NCrypta::TCryptaIdResolverMock>())};

        NSFStats::TStats stats{};
        NSFStats::TSolomonContext ctx{stats, {}};

        NBigRT::TRowsBatch rows;
        NBigRT::TRowMeta meta;
        rowsProcessor->Parse(ctx, meta, inputRows, rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 3);
        rowsProcessor->Process(ctx, TInstant::Now(), rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 3);

        const auto& msg{dynamic_cast<NCrypta::NEvent::TSoupEvent*>(rows[0].Message.Get())};
        UNIT_ASSERT_VALUES_EQUAL(TGenericID(msg->GetEdge().GetVertex1()).GetValue(), "abcd");
        UNIT_ASSERT_VALUES_EQUAL(TGenericID(msg->GetEdge().GetVertex2()).GetValue(), "1234567890");
    }

    Y_UNIT_TEST(TestSerialize) {
        const TString configJson{
            R"({
                "ChoiceType": "1",
                "MessageType": "SOUP",
                "Parser": {
                    "FromSoup": {
                        "B64Encoding": true,
                        "TimeStamp": "Unixtime"
                    }
                },
                "MaxOutputMessageSize": 220,
                "MaxOutputChunksCount": 3,
                "ReshardingModule": 3,
                "ShardingAlgorithm": "RoundRobin"
            })"};

        const TString inputRows{
            TStringBuilder{}
            << RowToProtoSoup("a1", "gaid", "1234567890", "yandexuid", 1567890120ull) << "\n"
            << RowToProtoSoup("a2", "idfa", "1234567810", "yandexuid", 1567890121ull) << "\n"
            << RowToProtoSoup("a3", "gaid", "1236543220", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a4", "idfa", "1236543230", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a5", "gaid", "1236543240", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a6", "idfa", "1236543250", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a7", "gaid", "1236543260", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a8", "idfa", "1236543270", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a9", "gaid", "1236543280", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("10", "idfa", "1236543211", "yandexuid", 1567890122ull)};

        const auto& config{NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson)};
        UNIT_ASSERT_VALUES_EQUAL(config.GetMaxOutputMessageSize(), 220);

        auto rowsProcessor{MakeRowsProcessor("resharder",
                                             config, 3, "someQueue", UINT64_MAX, MakeIntrusive<NCrypta::TCryptaIdResolverMock>())};

        NSFStats::TStats stats{};
        NSFStats::TSolomonContext ctx{stats, {}};

        NBigRT::TRowsBatch rows;
        NBigRT::TRowMeta meta;
        rowsProcessor->Parse(ctx, meta, inputRows, rows);
        UNIT_ASSERT_VALUES_EQUAL(10, rows.size());

        const size_t overflowSize{221};
        {
            auto fields{dynamic_cast<TRowFields*>(rows[9].Fields.Get())};
            fields->StandVersion = TString(overflowSize, '*');
        }

        rowsProcessor->Process(ctx, TInstant::Now(), rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 10);

        THashMap<TString, TVector<NBigRT::TAggregatedRow>> aggregatedRows;
        rowsProcessor->Aggregate(ctx, rows, aggregatedRows);
        UNIT_ASSERT_VALUES_EQUAL(aggregatedRows.size(), 1);
        for (const auto& [queue, aggregatedRowsForQueue] : aggregatedRows) {
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue.size(), 10);
        }

        THashMap<TString, TVector<NBigRT::TSerializedRow>> outputs{};
        rowsProcessor->Serialize(ctx, aggregatedRows, outputs);
        UNIT_ASSERT_VALUES_EQUAL(outputs.size(), 1);
        for (const auto& [queue, outputsForQueue] : outputs) {
            UNIT_ASSERT_VALUES_EQUAL(outputsForQueue.size(), 7);
        }

        MessageDifferencer protoDiff;
        protoDiff.set_message_field_comparison(MessageDifferencer::EQUIVALENT);
        protoDiff.set_repeated_field_comparison(MessageDifferencer::AS_SET);

        {
            size_t rowNum{0};
            for (const auto& [queue, outputsForQueue] : outputs) {
                for (auto& output : outputsForQueue) {
                    NCrypta::NEvent::TEventMessage message{};
                    TStringBuf skip;
                    size_t count{0};
                    for (NFraming::TUnpacker unpacker(output.Data);
                         unpacker.NextFrame(message, skip);
                         ++rowNum, ++count) {
                        UNIT_ASSERT(skip.empty());
                        UNIT_ASSERT_VALUES_EQUAL(aggregatedRows[queue][rowNum].Shard, output.Shard);
                        UNIT_ASSERT_VALUES_EQUAL(aggregatedRows[queue][rowNum].TimeStamp.TimeT(), message.GetTimeStamp());
                        auto body{NCrypta::NEvent::UnpackAny(message)};
                        TString diff{};
                        protoDiff.ReportDifferencesToString(&diff);
                        UNIT_ASSERT_C(protoDiff.Compare(*aggregatedRows[queue][rowNum].Message, *body), diff);
                    }
                    UNIT_ASSERT(count <= 4);
                }
            }
            UNIT_ASSERT_VALUES_EQUAL(rowNum, rows.size());
        }
    }

    Y_UNIT_TEST(TestSerializeClean) {
        const TString configJson{
            R"({
                "ChoiceType": "1",
                "MessageType": "SOUP",
                "Parser": {
                    "FromSoup": {
                        "B64Encoding": true,
                        "TimeStamp": "Unixtime"
                    }
                },
                "MaxOutputChunksCount": 3,
                "ReshardingModule": 3,
                "ShardingAlgorithm": "RoundRobin"
            })"};

        const TString inputRows{
            TStringBuilder{}
            << RowToProtoSoup("a1", "gaid", "1234567890", "yandexuid", 1567890120ull) << "\n"
            << RowToProtoSoup("a2", "idfa", "1234567810", "yandexuid", 1567890121ull) << "\n"
            << RowToProtoSoup("a3", "gaid", "1236543220", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a4", "idfa", "1236543230", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a5", "gaid", "1236543240", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a6", "idfa", "1236543250", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a7", "gaid", "1236543260", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a8", "idfa", "1236543270", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("a9", "gaid", "1236543280", "yandexuid", 1567890122ull) << "\n"
            << RowToProtoSoup("10", "idfa", "1236543211", "yandexuid", 1567890122ull)};

        auto rowsProcessor{MakeRowsProcessor("resharder",
                                             NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson),
                                             3, "someQueue", UINT64_MAX, MakeIntrusive<NCrypta::TCryptaIdResolverMock>())};

        NSFStats::TStats stats{};
        NSFStats::TSolomonContext ctx{stats, {}};

        NBigRT::TRowsBatch rows;
        NBigRT::TRowMeta meta;
        rowsProcessor->Parse(ctx, meta, inputRows, rows);
        rowsProcessor->Process(ctx, TInstant::Now(), rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 10);

        THashMap<TString, TVector<NBigRT::TAggregatedRow>> aggregatedRows;
        rowsProcessor->Aggregate(ctx, rows, aggregatedRows);
        UNIT_ASSERT_VALUES_EQUAL(aggregatedRows.size(), 1);
        for (const auto& [queue, aggregatedRowsForQueue] : aggregatedRows) {
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue.size(), 10);
        }

        THashMap<TString, TVector<NBigRT::TSerializedRow>> outputs{};
        rowsProcessor->Serialize(ctx, aggregatedRows, outputs);
        UNIT_ASSERT_VALUES_EQUAL(outputs.size(), 1);
        for (const auto& [queue, outputForQueue] : outputs) {
            UNIT_ASSERT_VALUES_EQUAL(outputForQueue.size(), 4);
        }

        aggregatedRows.clear();
        outputs.clear();
        rowsProcessor->Serialize(ctx, aggregatedRows, outputs);
        UNIT_ASSERT(outputs.empty());
    }

    Y_UNIT_TEST(TestProcessSoupFmt) {
        const TString configJson{
            R"({
                "ChoiceType": "1",
                "MessageType": "SOUP",
                "Parser": {
                    "FromSoup": {
                        "B64Encoding": true,
                        "TimeStamp": "Unixtime"
                    }
                },
                "ReshardingModule": 3,
                "ShardingAlgorithm": "CryptaId"
            })"};

        const TString inputRows{
            R"(Gj4KGggbUhYKFAjvn7ruqr2VzbQBELOLucHd0I4BEhwIBoIBFwoVCIurrK3+nuDyjAEQyuvNq8ev2YUHGBUgBSDo5476BQ==
               Gj4KGggbUhYKFAjunbruqr2VzbQBELOLucHd0I4BEhwICIIBFwoVCIurrK3+nuDyjAEQseTNq8ev2YUHGBUgBSDo5476BQ==
               GkoKEAgFQgwKBHNvbWUSBGUuY28SMgghugEtCisIovi72teu/KywARD9xta96pzepegBGOH769X94reldiD02vyJ7vHxi/YBGFggEyDo5476BQ==
               Gj4KGQgcYhUKEwjjt8qXk/vur4gBEIid5Z6qtxESHQgYigEYChYIoeDTo9DV7o2BARDUl82Z77POzqQBGBIgBSDo5476BQ==)"};

        auto rowsProcessor{MakeRowsProcessor("resharder",
                                             NProtobufJson::Json2Proto<TRowsProcessorConfig>(configJson),
                                             3, "someQueue", UINT64_MAX, MakeIntrusive<NCrypta::TCryptaIdResolverMock>())};

        NSFStats::TStats stats{};
        NSFStats::TSolomonContext ctx{stats, {}};

        NBigRT::TRowsBatch rows;
        NBigRT::TRowMeta meta;
        THashMap<TString, TVector<NBigRT::TAggregatedRow>> aggregatedRows;
        rowsProcessor->Parse(ctx, meta, inputRows, rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 4);
        rowsProcessor->Process(ctx, TInstant::Now(), rows);
        UNIT_ASSERT_VALUES_EQUAL(rows.size(), 4);
        rowsProcessor->Aggregate(ctx, rows, aggregatedRows);
        UNIT_ASSERT_VALUES_EQUAL(aggregatedRows.size(), 1);
        for (const auto& [queue, aggregatedRowsForQueue] : aggregatedRows) {
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue.size(), 4);
        }

        {
            auto fields{*dynamic_cast<TRowFields*>(rows[0].Fields.Get())};
            UNIT_ASSERT_VALUES_EQUAL(fields.CryptaId, 200);
        }

        const auto GetSourceType{
            [](const NCrypta::NGraphEngine::TEdgeBetween& soupEdge) {
                using namespace NCrypta::NSoup;
                return NSourceType::ESourceType_descriptor()
                    ->FindValueByNumber(static_cast<i32>(soupEdge.GetSourceType()))
                    ->options()
                    .GetExtension(Name);
            }};

        const auto GetLogSource{
            [](const NCrypta::NGraphEngine::TEdgeBetween& soupEdge) {
                using namespace NCrypta::NSoup;
                return NLogSource::ELogSourceType_descriptor()
                    ->FindValueByNumber(static_cast<i32>(soupEdge.GetLogSource()))
                    ->options()
                    .GetExtension(Name);
            }};

        {
            const auto& msg{dynamic_cast<NCrypta::NEvent::TSoupEvent*>(rows[0].Message.Get())};
            {
                const auto id{TGenericID{msg->GetEdge().GetVertex1()}};
                UNIT_ASSERT_VALUES_EQUAL(id.GetValue(), "00023a85-d82e-45b3-b49a-55eaadce8fef");
                UNIT_ASSERT_VALUES_EQUAL(id.GetTypeString(), "gaid");
            }
            {
                const auto id{TGenericID{msg->GetEdge().GetVertex2()}};
                UNIT_ASSERT_VALUES_EQUAL(id.GetValue(), "070b657c757375ca8ce580f7e5ab158b");
                UNIT_ASSERT_VALUES_EQUAL(id.GetTypeString(), "email_md5");
            }
            UNIT_ASSERT_VALUES_EQUAL(GetSourceType(msg->GetEdge()), "account-manager");
            UNIT_ASSERT_VALUES_EQUAL(GetLogSource(msg->GetEdge()), "mm");

            UNIT_ASSERT_VALUES_EQUAL(msg->GetCryptaId1(), 200);
            UNIT_ASSERT_VALUES_EQUAL(msg->GetCryptaId2(), 601);
        }
        {
            const auto& msg{dynamic_cast<NCrypta::NEvent::TSoupEvent*>(rows[1].Message.Get())};
            {
                const auto id{TGenericID{msg->GetEdge().GetVertex1()}};
                UNIT_ASSERT_VALUES_EQUAL(id.GetValue(), "00023a85-d82e-45b3-b49a-55eaadce8eee");
                UNIT_ASSERT_VALUES_EQUAL(id.GetTypeString(), "gaid");
            }
            {
                const auto id{TGenericID{msg->GetEdge().GetVertex2()}};
                UNIT_ASSERT_VALUES_EQUAL(id.GetValue(), "070b657c757372318ce580f7e5ab158b");
                UNIT_ASSERT_VALUES_EQUAL(id.GetTypeString(), "phone_md5");
            }
            UNIT_ASSERT_VALUES_EQUAL(GetSourceType(msg->GetEdge()), "account-manager");
            UNIT_ASSERT_VALUES_EQUAL(GetLogSource(msg->GetEdge()), "mm");

            UNIT_ASSERT_VALUES_EQUAL(msg->GetCryptaId1(), 0);
            UNIT_ASSERT_VALUES_EQUAL(msg->GetCryptaId2(), 501);
        }
        {
            const auto& msg{dynamic_cast<NCrypta::NEvent::TSoupEvent*>(rows[2].Message.Get())};
            {
                const auto id{TGenericID{msg->GetEdge().GetVertex1()}};
                UNIT_ASSERT_VALUES_EQUAL(id.GetValue(), "some@e.co");
                UNIT_ASSERT_VALUES_EQUAL(id.GetTypeString(), "email");
            }
            {
                const auto id{TGenericID{msg->GetEdge().GetVertex2()}};
                UNIT_ASSERT_VALUES_EQUAL(id.GetValue(), "f617c78ee13f2d74764adf17dabafde1e84b78e6a7b5a37db059f1757b4efc22");
                UNIT_ASSERT_VALUES_EQUAL(id.GetTypeString(), "email_sha256");
            }
            UNIT_ASSERT_VALUES_EQUAL(GetSourceType(msg->GetEdge()), "sha256");
            UNIT_ASSERT_VALUES_EQUAL(GetLogSource(msg->GetEdge()), "preproc");

            UNIT_ASSERT_VALUES_EQUAL(msg->GetCryptaId1(), 732);
            UNIT_ASSERT_VALUES_EQUAL(msg->GetCryptaId2(), 600);
        }
        {
            const auto& msg{dynamic_cast<NCrypta::NEvent::TSoupEvent*>(rows[3].Message.Get())};
            {
                const auto id{TGenericID{msg->GetEdge().GetVertex1()}};
                UNIT_ASSERT_VALUES_EQUAL(id.GetValue(), "000045BA-A3D9-4E88-885F-BBD932F29BE3");
                UNIT_ASSERT_VALUES_EQUAL(id.GetTypeString(), "idfa");
            }
            {
                const auto id{TGenericID{msg->GetEdge().GetVertex2()}};
                UNIT_ASSERT_VALUES_EQUAL(id.GetValue(), "a49d399ef3334bd4811bbaad0474f021");
                UNIT_ASSERT_VALUES_EQUAL(id.GetTypeString(), "mm_device_id");
            }
            UNIT_ASSERT_VALUES_EQUAL(GetSourceType(msg->GetEdge()), "app-metrica");
            UNIT_ASSERT_VALUES_EQUAL(GetLogSource(msg->GetEdge()), "mm");

            UNIT_ASSERT_VALUES_EQUAL(msg->GetCryptaId1(), 0);
            UNIT_ASSERT_VALUES_EQUAL(msg->GetCryptaId2(), 500);
        }

        {
            auto fields{*dynamic_cast<TRowFields*>(rows[0].Fields.Get())};
            UNIT_ASSERT_VALUES_EQUAL(fields.CryptaId, 200);
        }
        {
            auto fields{*dynamic_cast<TRowFields*>(rows[1].Fields.Get())};
            UNIT_ASSERT_VALUES_EQUAL(fields.CryptaId, 501);
        }
        {
            auto fields{*dynamic_cast<TRowFields*>(rows[2].Fields.Get())};
            UNIT_ASSERT_VALUES_EQUAL(fields.CryptaId, 732);
        }
        {
            auto fields{*dynamic_cast<TRowFields*>(rows[3].Fields.Get())};
            UNIT_ASSERT_VALUES_EQUAL(fields.CryptaId, 500);
        }

        UNIT_ASSERT_VALUES_EQUAL(rows[0].TimeStamp, TInstant::Seconds(1598272488ull));
        UNIT_ASSERT_VALUES_EQUAL(rows[1].TimeStamp, TInstant::Seconds(1598272488ull));
        UNIT_ASSERT_VALUES_EQUAL(rows[2].TimeStamp, TInstant::Seconds(1598272488ull));
        UNIT_ASSERT_VALUES_EQUAL(rows[3].TimeStamp, TInstant::Seconds(1598272488ull));

        for (const auto& [queue, aggregatedRowsForQueue] : aggregatedRows) {
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue[0].Shard, 200 % 3);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue[1].Shard, 501 % 3);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue[2].Shard, 732 % 3);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedRowsForQueue[3].Shard, 500 % 3);
        }
    }
}
