#include "decryptor.h"
#include <util/generic/yexception.h>
#include <util/string/util.h>

namespace NCrypta::NGraph {

TDecryptor::TDecryptor(const TString& rsaKey)
    : KeyBio(BIO_new_mem_buf(rsaKey.data(), -1))
    , Rsa(PEM_read_bio_RSAPrivateKey(KeyBio.Get(), NULL, NULL, NULL))
{
    Y_ENSURE(Rsa.Get() != NULL, "Invalid key: RSA context was not initialized");
}

TString TDecryptor::Decrypt(const TString& di) {
    static const int RSA_ENCRYPTED_SIZE = 128;
    static const int AES_KEY_SIZE = 16;
    static const int MAX_DATA_SIZE = 8192;
    static const int MIN_DATA_SIZE = 128;

    unsigned char key_data[32];
    unsigned char encrypted[4096];
    unsigned char plaintext[4096];

    if (di.size() > MAX_DATA_SIZE || di.size() <= MIN_DATA_SIZE) {
        ythrow yexception() << "Condition data length "  << MIN_DATA_SIZE
        << " < length <= " << MAX_DATA_SIZE << ", violated for: "  << di.size();
    }

    memcpy(encrypted, di.c_str(), di.size());
    auto result = RSA_private_decrypt(
        RSA_ENCRYPTED_SIZE,
        encrypted,
        key_data,
        Rsa.Get(),
        RSA_PKCS1_PADDING);
    Y_ENSURE(result != -1, "RSA_private_decrypt");

    TOpenSSLHolder<EVP_CIPHER_CTX> ctx(EVP_CIPHER_CTX_new());
    Y_ENSURE(ctx.Get() != NULL, "EVP_CIPHER_CTX_new");

    Y_ENSURE(EVP_DecryptInit_ex(ctx.Get(), EVP_aes_128_cbc(), nullptr, key_data, key_data + AES_KEY_SIZE) == 1, "EVP_DecryptInit_ex");
    auto cipher_len = di.size() - RSA_ENCRYPTED_SIZE;
    Y_ENSURE(cipher_len > 0, "Data to decrypt should have length > 0");
    int decrypted_len = cipher_len;
    Y_ENSURE(EVP_DecryptUpdate(ctx.Get(), plaintext, &decrypted_len, encrypted + RSA_ENCRYPTED_SIZE, cipher_len) == 1, "EVP_DecryptUpdate");
    int tail_len = cipher_len - decrypted_len;
    Y_ENSURE(EVP_DecryptFinal_ex(ctx.Get(), plaintext + decrypted_len, &tail_len) == 1, "EVP_DecryptFinal_ex");
    plaintext[decrypted_len + tail_len] = 0;

    return TString((char*)plaintext);
}

}
