#include "keyboards.h"

namespace {
    TString getTableModTime(TIntrusivePtr<NYT::IClient> client, const TString& tableName) {
        const static TString attrName = "modification_time";
        TString attrQuery = (TStringBuilder{} << tableName << "/@" << attrName);

        return client->Get(attrQuery).AsString();
    }
}

TUuidStorage::TUuidStorage(const TYtConfig& ytConfig, TRtdiOptions::TKeyboardsExperiment expOptions)
    : Enabled(expOptions.GetEnabled())
    , TableName(std::move(expOptions.GetTableName()))
    , ExperimentMask((1 << expOptions.GetExperimentBits()) - 1) // e.f. 3 = (1 << 3) - 1 = 111b
    , YtClient(NYT::CreateClient(ytConfig.GetProxy()))
    , StoragePtr(MakeHolder<TStorage>())
{
}

bool TUuidStorage::Update() {
    using namespace NYT;

    auto currentModificationTime = getTableModTime(YtClient, TableName);
    if (currentModificationTime == LastModificationTime) {
        return false;
    }
    LastModificationTime = std::move(currentModificationTime);

    auto newStorage = MakeHolder<TStorage>();
    auto ytReader = YtClient->CreateTableReader<TNode>(TableName);

    for (; ytReader->IsValid(); ytReader->Next()) {
        auto& row = ytReader->GetRow();
        auto& value = row["kuuid"].AsString();

        newStorage->insert(std::move(value));
    }

    {
        TWriteGuard writeGuard(AccessLock);
        std::swap(StoragePtr, newStorage);
    }

    return true;
}

bool TUuidStorage::CheckIfDrop(ui64 yuid, const TString& uuid) {
    if (Enabled && ((yuid & ExperimentMask) == 0)) {
        TReadGuard readGuard(AccessLock);
        return StoragePtr->contains(uuid);
    }

    return false;
}

size_t TUuidStorage::Size() {
    TReadGuard readGuard(AccessLock);
    return StoragePtr->size();
}

bool TUuidStorage::IsEnabled() {
    return Enabled;
}
