#include "lbwrapper.h"
#include <util/string/cast.h>

using NPersQueue::CreateTVMCredentialsProvider;
using NPersQueue::ICredentialsProvider;
using NPersQueue::ILogger;
using NPersQueue::TConsumerSettings;
using NPersQueue::TPQLibSettings;

using NCryptaLogbroker::CreateTvmProvider;
using NCryptaLogbroker::TLogbrokerPuller;

THolder<TLogbrokerPuller> CreateLogBrokerPuller(const TRtdiOptions& options, std::shared_ptr<NTvmAuth::TTvmClient> tvmClient) {
    TPQLibSettings pqSettings;
    TConsumerSettings consumerSettings;
    const auto& lboptions = options.GetLogbrokerOptions();

    /* pqlib */
    pqSettings.ThreadsCount = lboptions.GetThreadsCount();

    /* pq consumer */
    consumerSettings.Server = NPersQueue::TServerSetting{
        lboptions.GetServer(),
        static_cast<ui16>(lboptions.GetPort())};
    consumerSettings.Topics.push_back(lboptions.GetTopic());
    consumerSettings.ClientId = options.GetTvmOptions().GetClientAlias();
    consumerSettings.MaxInflyRequests = lboptions.GetMaxInflyRequests();
    consumerSettings.MaxMemoryUsage = lboptions.GetMaxMemoryUsage();
    consumerSettings.UseLockSession = lboptions.GetUseLockSession();
    consumerSettings.ReadMirroredPartitions = lboptions.GetReadMirroredPartitions();

    /* taken from kikimr example */
    consumerSettings.MaxCount = 0;
    consumerSettings.MaxSize = 10 << 20;
    consumerSettings.Unpack = true;
    consumerSettings.MaxUncommittedCount = 0;

    if (lboptions.GetSkipLog()) {
        consumerSettings.ReadTimestampMs = TInstant::Now().MilliSeconds() - 60 * 1000;
    } else {
        consumerSettings.ReadTimestampMs = 0;
    }

    TIntrusivePtr<ILogger> logger = MakeIntrusive<NPersQueue::TCerrLogger>(static_cast<int>(options.GetLogOptions().GetPQlibPriority()));

    /* setup TVM */
    auto alias = lboptions.GetServerStringTvmId();
    consumerSettings.CredentialsProvider = CreateTVMCredentialsProvider(tvmClient, logger, alias);

    TMaybe<char> splitter = lboptions.GetNoSplitter() ? Nothing() : TMaybe<char>{'\n'};
    return MakeHolder<TLogbrokerPuller>(pqSettings, consumerSettings, logger, NMonitoring::TLabels{}, splitter);
}

TLogbrokerPusher::TLogbrokerPusher(TLogbrokerPusherConfig options, std::shared_ptr<NTvmAuth::TTvmClient> tvmClient)
    : Options(std::move(options))
    , Logger(MakeIntrusive<NPersQueue::TCerrLogger>(Options.PQlibLogPriority))
    , PQ({.ThreadsCount = Options.ThreadsCount, .DefaultLogger = Logger})
{
    NPersQueue::TProducerSettings settings;

    settings.Server = NPersQueue::TServerSetting{
        Options.Server,
        Options.Port};
    settings.Topic = Options.Topic;
    settings.SourceId = Options.SourceId;

    settings.ReconnectOnFailure = true;
    settings.Codec = NPersQueueCommon::ECodec::GZIP;
    settings.CredentialsProvider = CreateTVMCredentialsProvider(tvmClient, Logger, Options.ServerStringTvmId);

    auto pg = Options.LockTo;
    if (pg > 0) {
        settings.PartitionGroup = pg;
    }

    Producer = PQ.CreateProducer(settings, Logger);
    auto future = Producer->Start();
    future.Wait();

    if (future.GetValue().Response.HasError()) {
        ythrow yexception() << "Unable to create Logbroker producer.";
    }
}

TLogbrokerPusher::EResult TLogbrokerPusher::Push(const TString& payload) {
    TInstant timestamp = TInstant::Now();

    auto seqNo = AtomicGetAndIncrement(Sequence);
    auto future = Producer->Write(seqNo, NPersQueue::TData{payload, timestamp});
    auto to = Options.TimeoutMs;

    if (to > 0) {
        future.Wait(TDuration::MilliSeconds(to));
    } else {
        future.Wait();
    }
    if (future.HasValue()) {
        if (future.GetValue().Response.HasError()) {
            return EResult::E_ERROR;
        }
        return EResult::E_OK;
    }

    return EResult::E_TIMEOUT;
}
