#include "rtdi.h"

#include <library/cpp/logger/global/global.h>

using namespace NRtdiConstants;

namespace {
    template <typename T>
    bool futureIsReady(const std::future<T>& f) {
        return f.wait_for(std::chrono::seconds(0)) == std::future_status::ready;
    }
}

bool TRtdi::PeriodicUpdateExperimentSystem() {
    if (!Options.GetExperimentOptions().GetEnabled()) {
        return true;
    }

    if (ExpSystemManager->Update()) {
        INFO_LOG << "ExperimentSystem updated." << Endl;
        return true;
    } else {
        ERROR_LOG << "Error fetching experiment system." << Endl;
    }

    return false;
}

bool TRtdi::PeriodicPrintMetrics() {
    Cout << TInstant::Now() << Endl;
    Cout << LogbrokerPullerHandle->GetMetrics().Textify() << Endl;
    Cout << Metrics.Textify() << Endl;

    return true;
}

bool TRtdi::PeriodicUpdateKeyboardUuids() {
    if (!KeyboardUuids.IsEnabled()) {
        return true;
    }

    INFO_LOG << "Start KeyboardUuids update." << Endl;
    if (KeyboardUuids.Update()) {
        INFO_LOG << "KeyboardUuids were updated." << Endl;
    } else {
        INFO_LOG << "KeyboardUuids table wasn't changed." << Endl;
    }
    return true;
}

void TRtdi::PeriodicThread() {
    TVector<TPeriodicTask> PeriodicTasks;

    PeriodicTasks.emplace_back(
        "PeriodicPrintMetrics",
        [this]() { return PeriodicPrintMetrics(); },
        TDuration::Seconds(10));
    PeriodicTasks.emplace_back(
        "PeriodicUpdateExperimentSystem",
        [this]() { return PeriodicUpdateExperimentSystem(); },
        TDuration::Seconds(900));
    PeriodicTasks.emplace_back(
        "PeriodicUpdateKeyboardUuids",
        [this]() { return PeriodicUpdateKeyboardUuids(); },
        TDuration::Seconds(180));

    while (IsRunning()) {
        auto now = TInstant::Now();

        for (auto& task : PeriodicTasks) {
            if (now - task.LastRun >= task.Period) {
                if (task.Result.valid() && !futureIsReady(task.Result)) {
                    throw yexception() << "Periodic task " << task.Id << " hasn't finish.";
                } else {
                    if (task.Result.valid()) {
                        if (task.Result.get()) {
                            task.LastSuccess = now;
                        };
                    }
                    INFO_LOG << "Executing: " << task.Id << ", last: " << task.LastRun << ", success: " << task.LastSuccess << "\n";
                    task.LastRun = now;
                    task.Result = std::async(task.What);
                }
            }
        }

        Metrics.SetIGauge(METRIC_UPLOADER_QUEUE_SIZE, UploaderQueue.Size());
        Metrics.SetIGauge(METRIC_UPLOADER2_QUEUE_SIZE, Uploader2Queue.Size());
        Metrics.SetIGauge(METRIC_LOGBROKER_QUEUE_SIZE, LogbrokerQueue.Size());
        Metrics.SetIGauge(METRIC_VULTURE_SEQ, LogbrokerPusherVulture.GetSeq());
        Metrics.SetIGauge(METRIC_KEYBOARDS_SIZE, KeyboardUuids.Size());

        Sleep(TDuration::Seconds(1));
    }

    for (auto& task : PeriodicTasks) {
        if (task.Result.valid()) {
            task.Result.wait();
        }
    }
}
