#include <chrono>

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/json/json_reader.h>

#include <contrib/libs/re2/re2/re2.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <util/string/util.h>
#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/lib/native/identifiers/lib/id_types/duid.h>
#include <crypta/lib/native/identifiers/lib/id_types/yandexuid.h>
#include <crypta/lib/native/identifiers/lib/id_types/mm_device_id.h>

#include <rtmapreduce/mrtasks/rtcrypta_geo/proto/messages.pb.h>
#include <rtmapreduce/protos/data.pb.h>
#include <library/cpp/protobuf/json/proto2json.h>
#include <crypta/graph/soup/config/proto/bigb.pb.h>

#include "rtdi.h"

using namespace NRtdiConstants;

bool TRtdi::IsUploadPrivateOk(EPrivatePolicy policy, const THttpSender::TMetrikaData& tm) {
    switch (policy) {
        case EPrivatePolicy::UPLOAD_ALL_PRIVATE:
            return true;
        case EPrivatePolicy::DROP_ALL_PRIVATE:
            return !tm.Private;
        case EPrivatePolicy::DROP_PRIVATE_BY_METRIKA:
            return !tm.PrivateFlag;
        case EPrivatePolicy::DROP_PRIVATE_BY_TS:
            return !tm.PrivateTs;
    }
}

bool TRtdi::CheckAndNormalize(THttpSender::TMetrikaData& tm) {
    if (NIdentifiers::TYandexuid yuid{tm.Yuid}; yuid.IsSignificant()) {
        tm.Yuid = yuid.Normalize();
        tm.Yuid64 = FromString<ui64>(tm.Yuid);
    } else {
        tm.Yuid.clear();
        tm.Yuid64 = 0;
    }

    if (NIdentifiers::TDuid duid{tm.Duid}; duid.IsSignificant()) {
        tm.Duid = duid.Normalize();
        tm.Duid64 = FromString<ui64>(tm.Duid);
    } else {
        tm.Duid.clear();
    }

    if (tm.Yuid.empty() && tm.Duid.empty()) {
        return false;
    }

    if (NIdentifiers::TMmDeviceId mmDeviceId{tm.MmDeviceId}; mmDeviceId.IsSignificant()) {
        tm.MmDeviceId = mmDeviceId.Normalize();
    } else {
        return false;
    }

    if (tm.IdType) {
        if (NIdentifiers::TGenericID deviceId{*tm.IdType, tm.DeviceId}; deviceId.IsSignificant()) {
            tm.DeviceId = deviceId.Normalize();
        } else {
            return false;
        }
    }

    return true;
}

bool TRtdi::ShouldUploadCommon(const THttpSender::TMetrikaData& tm) {
    const ui64 denominator = 60;
    ui64 rest = Options.GetSampleYuidBy();

    /* yuid sampling */
    if (rest) {
        if (!(tm.Yuid64 % denominator == rest)) {
            return false;
        }
    }

    /* rps limit */
    return RpsLimiter.OneMoreTimeIsPossible();
}

bool TRtdi::ShouldUploadV1(const THttpSender::TMetrikaData& tm) {
    if (tm.Yuid64 == 0) {
        return false;
    }

    if (!IsUploadPrivateOk(Options.GetUploaderOptions().GetPrivatePolicy(), tm)) {
        return false;
    }

    if (Options.GetExperimentOptions().GetEnabled()) {
        bool enabled = ExpSystemManager->GetBigbParameters(tm.Yuid64).GetCryptaRtdiSettings().GetEnableRealtime();

        if (!enabled) {
            Metrics.CounterIncrement(METRIC_DROPPED_BY_EXP_V1);
            return false;
        }
    }

    return ShouldUploadCommon(tm);
}

bool TRtdi::ShouldUploadV2(const THttpSender::TMetrikaData& tm) {
    if (tm.Yuid64 == 0) {
        return false;
    }

    if (!IsUploadPrivateOk(Options.GetUploader2Options().GetPrivatePolicy(), tm)) {
        return false;
    }

    if (Options.GetExperimentOptions().GetEnabled()) {
        bool enabled = ExpSystemManager->GetBigbParameters(tm.Yuid64).GetCryptaRtdiSettings().GetEnableRealtimeV2();

        if (!enabled) {
            Metrics.CounterIncrement(METRIC_DROPPED_BY_EXP_V2);
            return false;
        }
    }

    return ShouldUploadCommon(tm);
}

bool TRtdi::ShouldUploadVulture(const THttpSender::TMetrikaData& tm) {
    if (!IsUploadPrivateOk(Options.GetLogbrokerOptions().GetVulturePrivatePolicy(), tm)) {
        return false;
    }

    if (Options.GetExperimentOptions().GetEnabled()) {
        NExperiments::TUserIds id;

        if (tm.Yuid64 > 0) {
            id.SetUniqId(tm.Yuid64);
        } else {
            id.SetDuid(tm.Duid64);
        }

        bool enabled = ExpSystemManager->GetBigbParameters(id).GetCrypta().GetCryptaRealtimeMatching().GetSocketsVulture();

        if (!enabled) {
            Metrics.CounterIncrement(METRIC_DROPPED_BY_EXP_VULTURE);
            return false;
        }
    }

    return ShouldUploadCommon(tm);
}

bool TRtdi::DecryptBrowserInfo(const TBrowserInfo& browserInfo, THttpSender::TMetrikaData& metrikaData) {
    using namespace NCrypta::NIdentifiersProto;
    static const TString keyIdfa{"ifa"};
    static const TString keyGaid{"google_aid"};
    static const TString keyMmDeviceId{"device_id"};
    static const TString keyOaid{"huawei_aid"};
    static const TString keyUuid{"uuid"};

    try {
        auto deviceInfoEncrypted = browserInfo["di"];
        RemoveAll(deviceInfoEncrypted, '\n');

        auto deviceInfoJsoned = Decryptor.Decrypt(Base64Decode(deviceInfoEncrypted));
        TDeviceInfo deviceInfo = TDeviceInfo(deviceInfoJsoned);

        metrikaData.Uuid = deviceInfo[keyUuid];
        metrikaData.MmDeviceId = deviceInfo[keyMmDeviceId];

        if (!deviceInfo[keyGaid].empty()) {
            metrikaData.DeviceId = deviceInfo[keyGaid];
            metrikaData.IdType = NIdType::GAID;
            Metrics.CounterIncrement(METRIC_TOTAL_GAIDS);
        } else if (!deviceInfo[keyOaid].empty()) {
            metrikaData.DeviceId = deviceInfo[keyOaid];
            metrikaData.IdType = NIdType::OAID;
            Metrics.CounterIncrement(METRIC_TOTAL_OAIDS);
        } else if (!deviceInfo[keyIdfa].empty()) {
            metrikaData.DeviceId = deviceInfo[keyIdfa];
            metrikaData.IdType = NIdType::IDFA;
            Metrics.CounterIncrement(METRIC_TOTAL_IDFAS);
        } else {
            metrikaData.IdType = Nothing();
        }
    } catch (const yexception& e) {
        NOTICE_LOG << e.what() << "\n";
        return false;
    }

    return true;
}

bool TRtdi::UniqidWasChanged(const TBrowserInfo& browserInfo) {
    const auto& cy = browserInfo["cy"];

    return (cy == "2");
}

void TRtdi::CallBack(const TStringBuf data) {
    Metrics.CounterIncrement(METRIC_TOTAL_MESSAGES);

    TBrowserInfo browserInfo;
    THttpSender::TMetrikaData metrikaData;

    try {
        NJson::TJsonValue resp;

        NJson::ReadJsonTree(data, &resp);
        metrikaData.Yuid = resp["yandexuid"].GetString();
        metrikaData.TimeStamp = resp["timestamp"].GetUInteger();

        browserInfo = TBrowserInfo(resp["browserinfo"].GetString());
        metrikaData.PrivateFlag = (browserInfo["pri"] == "1");
        metrikaData.Duid = browserInfo["u"];
    } catch (const yexception&) {
        Metrics.CounterIncrement(METRIC_ERROR_INVALID_MESSAGE);
        return;
    }

    auto latency = TInstant::Now().TimeT() - metrikaData.TimeStamp;
    Metrics.HistogramRecord(METRIC_WATCHLOG_LAG, latency);
    Metrics.SetIGauge(METRIC_LAST_MESSAGE_LAG, latency);

    if (Options.GetDropAll()) {
        return;
    }

    if (!DecryptBrowserInfo(browserInfo, metrikaData)) {
        Metrics.CounterIncrement(METRIC_ERROR_DECRYPTION);
        return;
    }

    if (!CheckAndNormalize(metrikaData)) {
        Metrics.CounterIncrement(METRIC_INVALID_OR_BANNED);
        return;
    }

    Metrics.CounterIncrement(METRIC_TOTAL_GOOD_MESSAGES);

    if (metrikaData.PrivateFlag) {
        Metrics.CounterIncrement(METRIC_PRIVATE_COOKIES);
    }

    if (metrikaData.Yuid64) {
        auto yuid_latency = TInstant::Now().TimeT() - (metrikaData.Yuid64 % 10000000000ul);
        Metrics.HistogramRecord(METRIC_YANDEXUID_LAG, yuid_latency);
        if (yuid_latency < Options.GetIssueTimeThreshold()) {
            Metrics.CounterIncrement(METRIC_PRIVATE_BY_TS);
            metrikaData.PrivateTs = true;
        }
    } else {
        Metrics.CounterIncrement(METRIC_DUIDS_ONLY);
    }

    metrikaData.Private = metrikaData.PrivateTs || metrikaData.PrivateFlag;

    if (Options.GetDropOnCy()) {
        if (UniqidWasChanged(browserInfo)) {
            Metrics.CounterIncrement(METRIC_DROPPED_ON_CY);
            return;
        }
    }

    if (metrikaData.Yuid64 && KeyboardUuids.CheckIfDrop(metrikaData.Yuid64, metrikaData.Uuid)) {
        Metrics.CounterIncrement(METRIC_DROPPED_KEYBOARDS);
        return;
    }

    if (Options.GetUploaderOptions().GetEnabled() && ShouldUploadV1(metrikaData)) {
        auto unused = UploaderQueue.AddFunc([this, metrikaData]() {
            if (metrikaData.Private) {
                Metrics.CounterIncrement(METRIC_UPLOADER_PRIVATE);
            }

            auto start_time = TInstant::Now();
            auto result = Sender.DoSend(metrikaData);
            auto query_duration = TInstant::Now() - start_time;

            Metrics.HistogramRecord(METRIC_UPLOADER_REPLY_TIME, query_duration.MilliSeconds());

            switch (result) {
                case THttpSender::TReplyCode::OK:
                    Metrics.CounterIncrement(METRIC_UPLOADER_OK);
                    break;
                case THttpSender::TReplyCode::TIMEOUT:
                    Metrics.CounterIncrement(METRIC_UPLOADER_TIMEOUT);
                    break;
                case THttpSender::TReplyCode::ERROR:
                    Metrics.CounterIncrement(METRIC_UPLOADER_ERROR);
                    break;
            }
        });
        Y_UNUSED(unused);
    }

    if (Options.GetUploader2Options().GetEnabled() && ShouldUploadV2(metrikaData)) {
        auto unused = Uploader2Queue.AddFunc([this, metrikaData]() {
            auto start_time = TInstant::Now();
            auto result = Sender.DoSend2(metrikaData, GetIdserv2Ticket());
            auto query_duration = TInstant::Now() - start_time;

            Metrics.HistogramRecord(METRIC_UPLOADER2_REPLY_TIME, query_duration.MilliSeconds());

            switch (result) {
                case THttpSender::TReplyCode::OK:
                    Metrics.CounterIncrement(METRIC_UPLOADER2_OK);
                    break;
                case THttpSender::TReplyCode::TIMEOUT:
                    Metrics.CounterIncrement(METRIC_UPLOADER2_TIMEOUT);
                    break;
                case THttpSender::TReplyCode::ERROR:
                    Metrics.CounterIncrement(METRIC_UPLOADER2_ERROR);
                    break;
            }
        });
        Y_UNUSED(unused);
    }

    if (Options.GetLogbrokerOptions().GetVultureEnableUpload() && ShouldUploadVulture(metrikaData)) {
        using namespace NCrypta::NIdentifiersProto;
        using namespace NCrypta::NSoup;

        auto unused = LogbrokerQueue.AddFunc([this, metrikaData]() {
            Metrics.SetIGauge(METRIC_VULTURE_INFLY, LogbrokerPusherVulture.GetAndIncInFly());

            NBB::TLinks links;
            links.SetLogSource(NLogSource::ELogSourceType::WATCH_LOG);
            links.SetSourceType(NSourceType::ESourceType::RTDI);
            links.SetIndevice(true);
            links.SetLogEventTimestamp(metrikaData.TimeStamp);

            auto usage = Options.GetLogbrokerOptions().GetVultureUsage();
            if (usage != 0) {
                links.AddUsage(static_cast<NBB::EBbLinkUsage>(usage));
            }

            auto vertex = links.AddVertices();
            if (metrikaData.Yuid64) {
                if (metrikaData.Private) {
                    vertex->SetIdType(NIdType::PRIVATE_YANDEXUID);
                    Metrics.CounterIncrement(METRIC_VULTURE_PRIVATE);
                } else {
                    vertex->SetIdType(NIdType::YANDEXUID);
                }
                vertex->SetId(metrikaData.Yuid);
            } else {
                vertex->SetIdType(NIdType::DUID);
                vertex->SetId(metrikaData.Duid);
            }

            if (metrikaData.IdType) {
                vertex = links.AddVertices();
                vertex->SetId(metrikaData.DeviceId);
                vertex->SetIdType(*metrikaData.IdType);
            }

            vertex = links.AddVertices();
            vertex->SetId(metrikaData.MmDeviceId);
            vertex->SetIdType(NIdType::MM_DEVICE_ID);

            TString serialized;
            Y_PROTOBUF_SUPPRESS_NODISCARD links.SerializeToString(&serialized);

            auto result = LogbrokerPusherVulture.Push(serialized);
            switch (result) {
                case TLogbrokerPusher::EResult::E_OK:
                    Metrics.CounterIncrement(METRIC_VULTURE_UPLOAD_OK);
                    break;
                case TLogbrokerPusher::EResult::E_ERROR:
                    Metrics.CounterIncrement(METRIC_VULTURE_UPLOAD_ERR);
                    break;
                case TLogbrokerPusher::EResult::E_TIMEOUT:
                    Metrics.CounterIncrement(METRIC_VULTURE_UPLOAD_TO);
                    break;
            }

            LogbrokerPusherVulture.DecInfly();
        });
        Y_UNUSED(unused);
    }
}

void TRtdi::StreamThread() {
    auto needStop = [this]() {
        return !IsRunning();
    };
    auto callBack = [this](const TStringBuf data) {
        CallBack(data);
    };

    LogbrokerPullerHandle->Process(callBack, needStop);

    INFO_LOG << "Waiting UploaderQueue.." << Endl;
    UploaderQueue.Stop();
}

void TRtdi::ServeHttpRequest(const NNeh::IRequestRef& request) {
    Metrics.CounterIncrement(METRIC_HTTP_REQUESTS);

    try {
        TString json;
        if (request->Service() == "stat") {
            json = Metrics.Solomonify();
        }
        if (request->Service() == "lbstat") {
            json = LogbrokerPullerHandle->GetMetrics().Solomonify();
        }

        NNeh::TData reply(json.data(), json.data() + json.size());
        request->SendReply(reply);
    } catch (...) {
        NOTICE_LOG << CurrentExceptionMessage() << "\n";
        Metrics.CounterIncrement(METRIC_HTTP_ERRORS);
        request->SendError(NNeh::IRequest::BadRequest, "");
    }
}

void TRtdi::HttpServerThread() {
    using namespace NNeh;

    IServicesRef ssr = CreateLoop();

    ssr->Add(Options.GetHttpServerOptions().GetListenAt() + "stat", [this](const NNeh::IRequestRef& request) {
        ServeHttpRequest(request);
    });
    ssr->Add(Options.GetHttpServerOptions().GetListenAt() + "lbstat", [this](const NNeh::IRequestRef& request) {
        ServeHttpRequest(request);
    });
    ssr->ForkLoop(Options.GetHttpServerOptions().GetThreads());

    while (IsRunning()) {
        Sleep(TDuration::Seconds(1));
    }
}

void TRtdi::Stop() {
    if (!AtomicSwap(&Running, false)) {
        ERROR_LOG << "Stop called when already stopping." << Endl;
        return;
    }

    INFO_LOG << "Waiting HttpServerThread.." << Endl;
    HttpServerThreadHandle->Join();
    INFO_LOG << "Waiting PeriodicThread.." << Endl;
    PeriodicThreadHandle->Join();
    INFO_LOG << "Waiting StreamThread.." << Endl;
    StreamThreadHandle->Join();
}

void TRtdi::Start() {
    if (AtomicSwap(&Running, true)) {
        ERROR_LOG << "Start called while already running." << Endl;
        return;
    }

    INFO_LOG << "Starting StreamThread.." << Endl;
    StreamThreadHandle->Start();
    INFO_LOG << "Starting PeriodicThread.." << Endl;
    PeriodicThreadHandle->Start();
    INFO_LOG << "Starting HttpServerThread.." << Endl;
    HttpServerThreadHandle->Start();
}
