#pragma once

#include "browserinfo.h"
#include "keyboards.h"
#include "lbwrapper.h"
#include "uploader.h"
#include "constants.h"

#include <crypta/lib/native/metrics_helper/metrics_helper.h>
#include <crypta/graph/rtdi-rt/decryptor/decryptor.h>
#include <crypta/graph/rtdi-rt/proto/options.pb.h>

#include <ads/bsyeti/libs/experiments/plugins/bigb/plugin.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/neh/rpc.h>
#include <library/cpp/neh/neh.h>
#include <library/cpp/tvmauth/client/facade.h>
#include <util/generic/hash_set.h>
#include <util/generic/strbuf.h>
#include <util/generic/utility.h>
#include <util/datetime/base.h>
#include <util/system/thread.h>
#include <util/generic/set.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/thread/pool.h>
#include <crypta/lib/native/exp_system/exp_system.h>
#include <crypta/lib/native/rps_limiter/rps_limiter.h>

#include <ads/bsyeti/libs/experiments/plugins/bigb/plugin.h>

#include <functional>
#include <future>

using NCrypta::NGraph::TDecryptor;

/* this function makes it possible to create threads of member functons */
template <typename T, void (T::*M)()>
static void* HelperMemberFunc(void* arg) {
    T* obj = reinterpret_cast<T*>(arg);
    (obj->*M)();
    return nullptr;
}

struct TPeriodicTask {
    TString Id;
    std::function<bool()> What;
    TDuration Period;
    TInstant LastRun;
    TInstant LastSuccess;
    std::future<bool> Result;

    TPeriodicTask(const TString& id, std::function<bool()> what, TDuration period)
        : Id(id)
        , What(what)
        , Period(period)
        , LastRun(TInstant::Now())
        , LastSuccess(TInstant::Now())
    {
    }
};

class TRtdi {
private:
    using EMetricType = NCryptaSolomonMetrics::EMetricType;

    const NCryptaSolomonMetrics::TMetricsInit InitMetrics = {
        {NRtdiConstants::METRIC_UPLOADER_OK, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_UPLOADER_TIMEOUT, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_UPLOADER_ERROR, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_UPLOADER_PRIVATE, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_UPLOADER2_OK, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_UPLOADER2_TIMEOUT, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_UPLOADER2_ERROR, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_TOTAL_IDFAS, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_TOTAL_GAIDS, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_TOTAL_OAIDS, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_TOTAL_MESSAGES, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_ERROR_DECRYPTION, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_ERROR_INVALID_MESSAGE, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_DROPPED_BY_EXP_V1, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_DROPPED_BY_EXP_V2, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_DROPPED_BY_EXP_VULTURE, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_PRIVATE_BY_TS, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_HTTP_REQUESTS, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_HTTP_ERRORS, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_PRIVATE_COOKIES, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_TOTAL_GOOD_MESSAGES, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_VULTURE_UPLOAD_OK, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_VULTURE_UPLOAD_ERR, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_VULTURE_UPLOAD_TO, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_VULTURE_PRIVATE, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_VULTURE_INFLY, EMetricType::IGAUGE, {}},
        {NRtdiConstants::METRIC_VULTURE_SEQ, EMetricType::IGAUGE, {}},
        {NRtdiConstants::METRIC_LAST_MESSAGE_LAG, EMetricType::IGAUGE, {}},
        {NRtdiConstants::METRIC_UPLOADER_QUEUE_SIZE, EMetricType::IGAUGE, {}},
        {NRtdiConstants::METRIC_UPLOADER2_QUEUE_SIZE, EMetricType::IGAUGE, {}},
        {NRtdiConstants::METRIC_LOGBROKER_QUEUE_SIZE, EMetricType::IGAUGE, {}},
        {NRtdiConstants::METRIC_KEYBOARDS_SIZE, EMetricType::IGAUGE, {}},
        {NRtdiConstants::METRIC_INVALID_OR_BANNED, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_DROPPED_ON_CY, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_DROPPED_KEYBOARDS, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_DUIDS_ONLY, EMetricType::COUNTER, {}},
        {NRtdiConstants::METRIC_WATCHLOG_LAG, EMetricType::HISTOGRAM, {}, NMonitoring::TBucketBounds{1, 4, 10, 30, 60, 300, 600, 1800}},
        {NRtdiConstants::METRIC_YANDEXUID_LAG, EMetricType::HISTOGRAM, {}, NMonitoring::TBucketBounds{60 * 1, 60 * 10, 3600 * 1, 3600 * 12, 3600 * 24, 3600 * 24 * 2, 3600 * 24 * 7, 3600 * 24 * 30, 3600 * 24 * 365}},
        {NRtdiConstants::METRIC_UPLOADER_REPLY_TIME, EMetricType::HISTOGRAM, {}, NMonitoring::TBucketBounds{1, 2, 4, 8, 16, 32, 64, 99}},
        {NRtdiConstants::METRIC_UPLOADER2_REPLY_TIME, EMetricType::HISTOGRAM, {}, NMonitoring::TBucketBounds{1, 2, 4, 8, 16, 32, 64, 99}}};

    TRtdiOptions Options;
    TAutoPtr<TThread> StreamThreadHandle;
    TAutoPtr<TThread> PeriodicThreadHandle;
    TAutoPtr<TThread> HttpServerThreadHandle;
    NCryptaSolomonMetrics::TMetricsHelper Metrics;

    std::shared_ptr<NTvmAuth::TTvmClient> TvmClient;
    THolder<NCryptaLogbroker::TLogbrokerPuller> LogbrokerPullerHandle;
    TLogbrokerPusher LogbrokerPusherVulture;
    TAtomic Running;
    THolder<NCrypta::NExpSystem::TExperimentSystemManager> ExpSystemManager = nullptr;
    THttpSender Sender;
    TDecryptor Decryptor;
    TThreadPool UploaderQueue;
    TThreadPool Uploader2Queue;
    TThreadPool LogbrokerQueue;
    NCrypta::TRpsLimiter RpsLimiter;

    TUuidStorage KeyboardUuids;

    inline bool IsRunning() const {
        return AtomicGet(Running);
    }

    bool CheckAndNormalize(THttpSender::TMetrikaData& tm);
    bool UniqidWasChanged(const TBrowserInfo& browserInfo);
    bool ShouldUploadCommon(const THttpSender::TMetrikaData& tm);
    bool ShouldUploadV1(const THttpSender::TMetrikaData& tm);
    bool ShouldUploadV2(const THttpSender::TMetrikaData& tm);
    bool ShouldUploadVulture(const THttpSender::TMetrikaData& tm);
    bool IsUploadPrivateOk(EPrivatePolicy policy, const THttpSender::TMetrikaData& tm);

    void StreamThread();
    void CallBack(TStringBuf data);
    bool DecryptBrowserInfo(const TBrowserInfo&, THttpSender::TMetrikaData&);

    void PeriodicThread();
    bool PeriodicUpdateExperimentSystem();
    bool PeriodicPrintMetrics();
    bool PeriodicUpdateKeyboardUuids();

    void HttpServerThread();
    void ServeHttpRequest(const NNeh::IRequestRef&);

    NMonitoring::TLabels GetCommonLabels(const TRtdiOptions& options) const {
        NMonitoring::TLabels result;

        for (size_t i = 0; i < options.SolomonLabelsSize(); i++) {
            TVector<TString> res;
            Split(options.GetSolomonLabels(i), "=", res);
            result.Add(res[0], res[1]);
        }

        return result;
    }

    std::shared_ptr<NTvmAuth::TTvmClient> CreateTVMClient(TRtdiOptions& options) {
        NTvmAuth::NTvmApi::TClientSettings settings;

        options.MutableLogbrokerOptions()->SetServerStringTvmId(ToString(options.GetLogbrokerOptions().GetServerTvmId()));
        options.MutableUploader2Options()->SetServerStringTvmId(ToString(options.GetUploader2Options().GetServerTvmId()));

        settings.SetSelfTvmId(options.GetTvmOptions().GetClientTvmId());
        settings.EnableServiceTicketChecking();
        settings.EnableUserTicketChecking(NTvmAuth::EBlackboxEnv::ProdYateam);
        settings.EnableServiceTicketsFetchOptions(options.GetTvmOptions().GetClientTvmSecret(), {{options.GetLogbrokerOptions().GetServerStringTvmId(), options.GetLogbrokerOptions().GetServerTvmId()},
                                                                                                 {options.GetUploader2Options().GetServerStringTvmId(), options.GetUploader2Options().GetServerTvmId()}});

        NTvmAuth::TLoggerPtr logger = MakeIntrusive<NTvmAuth::TCerrLogger>(options.GetLogOptions().GetTvmPriority());
        return std::make_shared<NTvmAuth::TTvmClient>(settings, logger);
    }

    TString GetIdserv2Ticket() {
        return "X-Ya-Service-Ticket: " + TvmClient->GetServiceTicketFor(Options.GetUploader2Options().GetServerStringTvmId()) + "\r\n";
    }

    TLogbrokerPusher::TLogbrokerPusherConfig MakeVultureUploaderOptions(const TRtdiOptions& options) {
        TLogbrokerPusher::TLogbrokerPusherConfig config;

        config.ThreadsCount = options.GetLogbrokerOptions().GetThreadsCount();
        config.Server = options.GetLogbrokerOptions().GetServer();
        config.Port = static_cast<ui16>(options.GetLogbrokerOptions().GetPort());
        config.Topic = Options.GetLogbrokerOptions().GetVultureExportTopic();
        config.ServerStringTvmId = options.GetLogbrokerOptions().GetServerStringTvmId();
        config.LockTo = 0;
        config.TimeoutMs = options.GetLogbrokerOptions().GetVultureUploadTimeoutMs();
        config.PQlibLogPriority = static_cast<int>(options.GetLogOptions().GetPQlibPriority());
        config.SourceId = "vulture" + ToString(TInstant::Now().MilliSeconds());

        return config;
    }

public:
    explicit TRtdi(const TRtdiOptions& options)
        : Options(options)
        , Metrics(GetCommonLabels(Options), InitMetrics)
        , TvmClient(CreateTVMClient(Options))
        , LogbrokerPullerHandle(CreateLogBrokerPuller(Options, TvmClient))
        , LogbrokerPusherVulture(MakeVultureUploaderOptions(Options), TvmClient)
        , Running(false)
        , Sender(Options)
        , Decryptor(Options.GetMetrikaKey())
        , UploaderQueue(TThreadPool::TParams().SetBlocking(true).SetCatching(true))
        , Uploader2Queue(TThreadPool::TParams().SetBlocking(true).SetCatching(true))
        , LogbrokerQueue(TThreadPool::TParams().SetBlocking(true).SetCatching(true))
        , RpsLimiter(Options.GetRpsLimit())
        , KeyboardUuids(Options.GetYt(), Options.GetKeyboardOptions())
    {
        PeriodicUpdateKeyboardUuids();

        StreamThreadHandle = new TThread(HelperMemberFunc<TRtdi, &TRtdi::StreamThread>, this);
        PeriodicThreadHandle = new TThread(HelperMemberFunc<TRtdi, &TRtdi::PeriodicThread>, this);
        HttpServerThreadHandle = new TThread(HelperMemberFunc<TRtdi, &TRtdi::HttpServerThread>, this);

        if (Options.GetExperimentOptions().GetEnabled()) {
            DEBUG_LOG << "Fetching ExperimentSystem.."
                      << "\n";
            ExpSystemManager = MakeHolder<NCrypta::NExpSystem::TExperimentSystemManager>();
        }

        // Queue size should be large enough to deal with bursts after restart
        auto max_queue_size = Min(Options.GetUploaderOptions().GetMaxWorkersQueueSize(), Options.GetUploaderOptions().GetWorkers() * 10000);
        UploaderQueue.Start(Options.GetUploaderOptions().GetWorkers(), max_queue_size);

        max_queue_size = Min(Options.GetUploader2Options().GetMaxWorkersQueueSize(), Options.GetUploader2Options().GetWorkers() * 10000);
        Uploader2Queue.Start(Options.GetUploader2Options().GetWorkers(), max_queue_size);

        LogbrokerQueue.Start(Options.GetLogbrokerOptions().GetUploaderWorkers(), Options.GetLogbrokerOptions().GetUploaderWorkers() * 10);
    }

    ~TRtdi() {
        Stop();
    }

    void Start();
    void Stop();
};
