#include "uploader.h"
#include <library/cpp/neh/http_common.h>

TString THttpSender::buildQueryWithParams(const TString& url, const THttpParams& params) const {
    TVector<TString> pairs;

    for (const auto& it : params) {
        pairs.push_back(it.first + "=" + it.second);
    }

    return TStringBuilder() << url << "?" << JoinStrings(pairs, "&");
}

THttpSender::TReplyCode THttpSender::DoRequest(const TString& query, const TDuration timeout, TString header = "") {
    NNeh::TMessage message = {query, ""};
    if (!header.empty()) {
        if (!NNeh::NHttp::MakeFullRequest(message, header, "", "")) {
            return TReplyCode::ERROR;
        }
    }
    auto resp = NNeh::Request(message)->Wait(timeout);
    if (!resp) {
        return TReplyCode::TIMEOUT;
    } else {
        if (resp->IsError()) {
            return TReplyCode::ERROR;
        }
    }
    return TReplyCode::OK;
}

TString THttpSender::buildDevInfoQuery(const TMetrikaData& tm) const {
    using namespace NCrypta::NIdentifiersProto;
    auto params = CommonParams1;
    params["body.type"] = "DEV_INFO";
    params["body.dev_info.yuid"] = tm.Yuid;

    switch (*tm.IdType) {
        case NIdType::GAID: {
            params["body.dev_info.google_ad_id"] = tm.DeviceId;
            params["body.dev_info.device_id"] = tm.DeviceId;
            break;
        }
        case NIdType::OAID: {
            params["body.dev_info.oaid"] = tm.DeviceId;
            params["body.dev_info.device_id"] = tm.DeviceId;
            break;
        }
        case NIdType::IDFA: {
            params["body.dev_info.idfa"] = tm.DeviceId;
            params["body.dev_info.device_id"] = tm.DeviceId;
            break;
        }
        default: {
            ythrow yexception() << "incorrect MetrikaData IdType";
        }
    }

    return buildQueryWithParams(Options1.GetUrl(), params);
}

TString THttpSender::buildYuidDevIdQuery(const TMetrikaData& tm) const {
    auto params = CommonParams1;
    params["body.type"] = "YUID_DEVID";
    params["body.yuid_devid.yuid"] = tm.Yuid;
    params["body.yuid_devid.device_id"] = tm.DeviceId;

    return buildQueryWithParams(Options1.GetUrl(), params);
}

TString THttpSender::buildYuidDevIdQuery2(const TMetrikaData& tm) const {
    using namespace NCrypta::NIdentifiersProto;
    auto params = CommonParams2;

    if (tm.Private) {
        params["private_yandexuid"] = tm.Yuid;
    } else {
        params["yandexuid"] = tm.Yuid;
    }

    switch (*tm.IdType) {
        case NIdType::GAID: {
            params["gaid"] = tm.DeviceId;
            break;
        }
        case NIdType::OAID: {
            params["oaid"] = tm.DeviceId;
            break;
        }
        case NIdType::IDFA: {
            params["idfa"] = tm.DeviceId;
            break;
        }
        default: {
            ythrow yexception() << "incorrect MetrikaData IdType";
        }
    }

    return buildQueryWithParams(Options2.GetUrl(), params);
}

THttpSender::TReplyCode THttpSender::DoSend(const THttpSender::TMetrikaData& tm) {
    TReplyCode retval;
    auto timeout = TDuration::MilliSeconds(Options1.GetTimeoutMs());

    retval = DoRequest(buildDevInfoQuery(tm), timeout);
    if (retval != TReplyCode::OK) {
        return retval;
    }
    retval = DoRequest(buildYuidDevIdQuery(tm), timeout);
    return retval;
}

THttpSender::TReplyCode THttpSender::DoSend2(const THttpSender::TMetrikaData& tm, TString header) {
    auto timeout = TDuration::MilliSeconds(Options2.GetTimeoutMs());
    return DoRequest(buildYuidDevIdQuery2(tm), timeout, header);
}
