#include "bs_hit_event.h"
#include <crypta/graph/rtmr/lib/common/deduplicator_indexes.h>

#include <crypta/lib/native/proto_serializer/proto_serializer.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/proto/ext_fp/fp_event.pb.h>
#include <crypta/graph/rtmr/lib/common/get_normalized_host.h>
#include <crypta/graph/rtmr/lib/common/get_source_by_ip.h>
#include <crypta/graph/rtmr/lib/common/normalize_ip.h>
#include <crypta/graph/rtmr/lib/common/validate.h>
#include <crypta/graph/rtmr/proto/parsed_bs_watch_row.pb.h>
#include <util/digest/city.h>
#include <util/string/builder.h>
#include <util/string/cast.h>

namespace {
    TString GetTrait(const uatraits::detector::result_type& traits, const std::string& trait) {
        const auto& it = traits.find(trait);
        return it == traits.end() ? TString{} : TString(it->second);
    }
}

namespace NCrypta::NGraph {
    THitEventMapper::THitEventMapper(const TGeobaseLookup& geoData)
        : GeoData(geoData)
    {
    }

    TEventMapper::TEventMapper(const TGeobaseLookup& geoData)
        : THitEventMapper(geoData)
    {
    }

    THitMapper::THitMapper(const TResources& resources, const TGeobaseLookup& geoData)
        : THitEventMapper(geoData)
        , Resources(resources)
        , ResourceHolder(Resources.GetHolder())
        , Detector(ResourceHolder.GetUatraitsDetector())
    {
    }

    THitlogIdReducer::THitlogIdReducer(const TResources& resources, const TGeobaseLookup& geoData)
        : Resources(resources)
        , GeoData(geoData)
    {
    }

    void TEventMapper::DoParsed(const NYT::TNode& parsed, TWriter* writer) {
        static const TString counterClick = "2";

        const auto& duid = GetSafe<TString>(parsed, "duid");
        const auto& clientAddress = GetSafe<TString>(parsed, "clientip6");
        const auto& clientPort = GetSafe<TString>(parsed, "clientport");
        const auto& hitlogId = GetSafe<TString>(parsed, "hitlogid");
        const auto& timeStamp = GetSafe<TString>(parsed, "unixtime");
        const auto& regionId = GetSafe<TString>(parsed, "regionid");
        const auto& yuid = GetSafe<TString>(parsed, "rawuniqid");

        const auto& yclid = GetSafe<TString>(parsed, "logid");
        const auto& trackid = GetSafe<TString>(parsed, "trackid");
        const auto& cryptaid = GetSafe<TString>(parsed, "cryptaid");
        const auto& counterType = GetSafe<TString>(parsed, "countertype");

        try {
            if (!IsGDPR(GeoData, regionId)) {
                THitEventData message;
                if (ValidateYandexuid(yuid)) {
                    message.SetYuid(yuid);

                    if (ValidateYClid(yclid)
                        && ValidateYClid(trackid) // should have the same format
                        && (counterType == counterClick)
                        && (cryptaid != yuid))
                    {
                        TParsedBsWatchRow yclidMessage;
                        yclidMessage.SetYuid(yuid);
                        yclidMessage.SetTimestamp(FromString<i64>(timeStamp));
                        auto message = NCrypta::NProtoSerializer::ToString(yclidMessage);

                        writer->AddRow({yclid, timeStamp, message}, EOutputTo::YClidJoin);
                        writer->AddRow({trackid, timeStamp, message}, EOutputTo::YClidJoin);
                    }
                }

                if (ValidateDomainUserID(duid)
                    && ValidateHitlogId(hitlogId)
                    && ValidateAddress6(clientAddress)
                    && !IsPrivateRelayIp(GeoData, clientAddress)
                ) {
                    message.SetDuid(duid);
                    message.SetClientAddress(clientAddress);
                    message.SetClientPort(clientPort);
                    message.SetEventTime(FromString<ui64>(timeStamp));
                    message.SetFromEvent(true);

                    writer->AddRow({hitlogId, timeStamp, NCrypta::NProtoSerializer::ToString(message)}, EOutputTo::HitlogIdJoin);
                }
            }
        } catch (const yexception& e) {
            writer->AddRow({hitlogId, timeStamp, e.what()}, EOutputTo::Errors);
        }
    }

    void THitMapper::DoParsed(const NYT::TNode& parsed, TWriter* writer) {
        const auto& hitlogId = GetSafe<TString>(parsed, "hitlogid");
        const auto& referer = GetSafe<TString>(parsed, "referer");
        const auto& regionId = GetSafe<TString>(parsed, "regionid");
        const auto& timeStamp = GetSafe<TString>(parsed, "unixtime");
        const auto& userAgent = GetSafe<TString>(parsed, "useragent");
        const auto& clientip6 = GetSafe<TString>(parsed, "clientip6");

        const auto& host = GetNormalizedHost(referer);
        const auto& traits = Detector.detect(userAgent);

        try {
            if (!ValidateUATraits(traits) || IsGDPR(GeoData, regionId) || IsPrivateRelayIp(GeoData, clientip6) || !host) {
                return;
            }

            THitEventData message;
            message.SetOSVersion(GetTrait(traits, "OSVersion"));
            message.SetOSFamily(GetTrait(traits, "OSFamily"));
            message.SetBrowserName(GetTrait(traits, "BrowserName"));
            message.SetBrowserVersion(GetTrait(traits, "BrowserVersion"));
            message.SetDomain(*host);
            message.SetOriginalDomain(*host);
            message.SetUserAgent(userAgent);
            message.SetFromHit(true);

            writer->AddRow({hitlogId, timeStamp, NCrypta::NProtoSerializer::ToString(message)}, EOutputTo::HitlogIdJoin);

        } catch (const yexception& e) {
            writer->AddRow({hitlogId, timeStamp, e.what()}, EOutputTo::Errors);
        }
    }

    TString THitlogIdReducer::GetFingerprint(const THitEventData& data){
        auto digest = CityHash64(TStringBuilder() << data.GetUserAgent()
            << data.GetClientAddress()
            << data.GetClientPort()
        );

        return ToString(digest);
    }

    void THitlogIdReducer::Do(TReader* reader, TWriter* writer) {
        TShiftedClock::FreezeTimestampFromEnv();

        TResources::TResourcesHolder holder(Resources.GetHolder());
        const auto& sources = holder.GetSources();

        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();

            try {
                THitEventData input;
                Y_PROTOBUF_SUPPRESS_NODISCARD input.ParseFromString(TString{row.Value});

                State.MergeFrom(input);

                if (State.GetFromEvent() && State.GetFromHit()) {
                    if (ValidatePort(State.GetClientPort())) {
                        TParsedBsWatchRow out;
                        out.SetFpc(State.GetDuid());
                        out.SetOSVersion(State.GetOSVersion());
                        out.SetOSFamily(State.GetOSFamily());
                        out.SetOriginalDomain(State.GetDomain());
                        out.SetDomain(State.GetDomain());
                        out.SetBrowserName(State.GetBrowserName());
                        out.SetBrowserVersion(State.GetBrowserVersion());
                        out.SetTimestamp(TShiftedClock::Now().Seconds());
                        out.SetYuid(State.GetYuid());
                        out.SetSource("port");

                        const auto& fingerprint = GetFingerprint(State);
                        writer->AddRow({fingerprint, row.SubKey, NCrypta::NProtoSerializer::ToString(out)}, EOutputTo::Join);
                    }

                    TStringBuf ip{State.GetClientAddress()};
                    const auto fpSource = FindFirstPassing(sources, ip, GetSourceByIpSafe(GeoData, ip));

                    if (ui64 key; fpSource && TryFromString<ui64>(row.Key, key)) {
                        NExtFp::TFpEvent event;
                        const static TString sourcePrefix{"hitlog:"};

                        event.SetLogType("bs-event-log");
                        event.SetHitLogId(key);
                        event.SetIp(NormalizeIp(ip).data());
                        event.SetPort(FromString<ui32>(State.GetClientPort()));
                        event.SetDuid(FromString<ui64>(State.GetDuid()));
                        if (!State.GetYuid().empty()) { event.SetYuid(FromString<ui64>(State.GetYuid())); }
                        event.SetUnixtime(State.GetEventTime());
                        event.SetUserAgent(State.GetUserAgent());
                        event.SetDomain(State.GetDomain());
                        event.SetCurrentTimestamp(TShiftedClock::Now().Seconds());
                        event.SetSourceId(fpSource->GetSource());

                        const auto outputTableIndex = ToString(static_cast<int>(fpSource->IsDelayed()
                            ? EDeduplicatorTableIndex::ToExtFpDelayed
                            : EDeduplicatorTableIndex::ToExtFp));
                        writer->AddRow({sourcePrefix + State.GetDuid(), outputTableIndex, NProtoSerializer::ToJson(event)},
                            EOutputTo::ToDeduplicator);
                    }
                }
            } catch (const yexception& e) {
                writer->AddRow({row.Key, row.SubKey, e.what()}, EOutputTo::Errors);
            }
        }
    }
}
