#include "browserinfo.h"
#include "get_source_by_ip.h"
#include "validate.h"

#include "get_normalized_host.h"

#include <contrib/libs/re2/re2/re2.h>

#include <util/generic/hash_set.h>

using namespace NCrypta;
using namespace NCrypta::NGraph;

namespace NCrypta::NGraph {
    namespace {
        const char* const IS_EU = "is_eu";

        bool HasTrait(const uatraits::detector::result_type& traits, const uatraits::detector::result_type::key_type& key) {
            auto it = traits.find(key);
            return it != traits.end() && it->second == "true";
        }
    }

    bool ValidateYClid(const TString& yclid) {
        static const re2::RE2 yClidMatch(R"(^[0-9]{12,}$)");

        return re2::RE2::FullMatch(yclid, yClidMatch);
    }

    bool ValidateHitlogId(const TString& hitlogId) {
        static const re2::RE2 hitlogIdMatch(R"(^[0-9]{18,20}$)");

        return re2::RE2::FullMatch(hitlogId, hitlogIdMatch);
    }

    bool ValidateBrowserInfo(const TString& browserInfo) {
        static const TString urlWasReplaced("2");
        static const TString refererAndUrlWereReplaced("3");

        static const TString forceUrl("fu");
        static const TString externalLink("ln");
        static const TString downloadLink("dl");

        static const THashSet<TString> badForceUrlValues({urlWasReplaced,
                                                          refererAndUrlWereReplaced});

        return !badForceUrlValues.contains(NBrowserInfo::GetValue(browserInfo, forceUrl)) &&
               NBrowserInfo::GetValue(browserInfo, externalLink).empty() &&
               NBrowserInfo::GetValue(browserInfo, downloadLink).empty();
    }

    bool ValidateCounterID(const TString& counterID) {
        static const TString technicalCounter = "3";
        static const TString malwareCounter = "48547472";

        return (counterID != technicalCounter) && (counterID != malwareCounter);
    }

    bool ShouldSaveYuid(const TString& browserInfo) {
        static const TString cyKey{"cy"};

        if (browserInfo.empty()) {
            return false;
        }

        const auto& cy = NBrowserInfo::GetValue(browserInfo, cyKey);
        if ((cy == "1") || (cy == "2")) {
            return false;
        }

        return true;
    }

    bool ValidateDomainUserID(const TString& domainUserID) {
        return IsValidIdentifier<NIdentifiers::TDuid>(domainUserID);
    }

    bool ValidateSslSessionTicketIV(const TString& sessionTicket) {
        static const re2::RE2 sslSessionTicketMatch(R"(^[0-9a-fA-F]{32}$)");
        return re2::RE2::FullMatch(sessionTicket, sslSessionTicketMatch);
    }

    bool IsYandexDomain(const TString& domain) {
        static const re2::RE2 yndxRegex{
            R"(^(.*\.)?((ya.ru)|(yandex\.(com\.)?(ru|com|by|kz|md|am|az|ee|fr|ge|kg|lt|lv|tj|tm|tr|ua|uz)))$)"
        };
        return re2::RE2::FullMatch(domain, yndxRegex);
    }

    bool IsYastaticDomain(const TString& domain) {
        return domain.EndsWith("yastatic.net");
    }

    bool ValidateYandexReferer(const TString& referer, const TMaybe<TString>& originalDomain) {
        const auto& refererDomain = GetNormalizedHost(referer);
        return refererDomain.Defined() && originalDomain.Defined() && refererDomain != originalDomain && IsYandexDomain(*refererDomain);
    }

    bool ValidateYandexuid(const TString& yandexuid) {
        return IsValidIdentifier<NIdentifiers::TYandexuid>(yandexuid);
    }

    bool ValidateUATraits(const uatraits::detector::result_type& traits) {
        return !HasTrait(traits, "inAppBrowser") &&
               !HasTrait(traits, "isRobot") &&
               (HasTrait(traits, "ITP") || HasTrait(traits, "ITPMaybe"));
    }

    bool IsCorrectDomain(const TMaybe<TString>& domain) {
        return domain && IsCorrectDomain(*domain);
    }

    bool IsCorrectDomain(const TString& domain) {
        static const re2::RE2 domainMatch(R"(^[a-zA-Z0-9._-]+\.[a-zA-Z0-9-]{2,}$)");
        return re2::RE2::FullMatch(domain, domainMatch);
    }

    bool ValidateDomain(const TString& domain) {
        return !IsYastaticDomain(domain) && !IsYandexDomain(domain) && IsCorrectDomain(domain);
    }

    bool ValidateDomain(const TMaybe<TString>& domain) {
        return domain.Defined() && ValidateDomain(*domain);
    }

    bool IsGDPR(const NGeobase::TLookup& geoData, TGeoRegion regionId) {
        const auto& region = geoData.GetRegionById(regionId);
        return region.IsFieldExists(IS_EU) && region.GetBoolField(IS_EU) || regionId == 84;
    }

    bool IsGDPR(const NGeobase::TLookup& geoData, const TString& regionId) {
        TGeoRegion geoRegionId;

        if (TryFromString<TGeoRegion>(regionId, geoRegionId)) {
            return IsGDPR(geoData, geoRegionId);
        }

        return true;
    }

    bool IsPrivateRelayIp(const NGeobase::TLookup& geoData, const TString& ipAddress) {
        static const TString PrivateRelayISP{"icloud private relay"};

        return GetSourceByIpSafe(geoData, ipAddress) == PrivateRelayISP;
    }

    bool IsRostelecomExtfp(const TString& browserInfo) {
        static const TString rtKey{"rt"};

        const auto& rt = NBrowserInfo::GetValue(browserInfo, rtKey);

        return rt == "1";
    }

    bool ValidateAddress(const TString& address) {
        return ValidateAddress4(address) || ValidateAddress6(address);
    }

    bool ValidateAddress4(const TString& address) {
        static const re2::RE2 ipv4Match(R"(^(?:[0-9]{1,3}\.){3}[0-9]{1,3}$)");
        return re2::RE2::FullMatch(address, ipv4Match);
    }

    bool ValidateAddress6(const TString& address) {
        static const re2::RE2 ipv6Match(R"(^[0-9a-f:\.]+$)");
        return re2::RE2::FullMatch(address, ipv6Match);
    }

    bool ValidatePort(const TString& port) {
        static const re2::RE2 portMatch(R"(^[0-9]{4,5}$)");
        return re2::RE2::FullMatch(port, portMatch);
    }
}
