#include "join_fingerprints.h"

#include <crypta/graph/rtmr/lib/common/serialize_fingerprint_match.h>

#include <crypta/lib/native/time/shifted_clock.h>

#include <google/protobuf/util/message_differencer.h>

#include <util/generic/ymath.h>

using namespace NCrypta::NGraph;

TJoinFingerprints::TJoinFingerprints(const TResources& resources)
    : Resources(resources) {
}

void TJoinFingerprints::Do(TReader* reader, TWriter* writer) {
    time_t maxTimestamp = 0;
    TShiftedClock::FreezeTimestampFromEnv();

    for (; reader->IsValid(); reader->Next()) {
        const auto& row = reader->GetRow();

        try {
            const auto& timestamp = FromString<time_t>(row.SubKey);
            maxTimestamp = timestamp;

            if (!State.HasFingerprint()) {
                State.SetFingerprint(TString{row.Key});
            }

            if (reader->GetTableIndex() == EInputTable::RedirLog) {
                Y_PROTOBUF_SUPPRESS_NODISCARD YuidMessage.ParseFromString(TString{row.Value});
                NewMatchGetter.AddYuidEvent(std::move(YuidMessage));
            } else if (reader->GetTableIndex() == EInputTable::BsWatchLog) {
                Y_PROTOBUF_SUPPRESS_NODISCARD BsWatchRow.ParseFromString(TString{row.Value});
                NewMatchGetter.AddBsWatchEvent(std::move(BsWatchRow));
            } else {
                ythrow yexception() << "Unknown table index: " << reader->GetTableIndex();
            }
        } catch (const yexception& e) {
            writer->AddRow({row.Key, row.SubKey, e.what()}, Errors);
        }
    }

    auto resourceHolder = Resources.GetHolder();
    const auto& config = resourceHolder.GetConfig();

    try {
        const auto& newReports = NewMatchGetter.GetNewReports(State, maxTimestamp, config);

        for (const auto& [yuidMessage, bsWatchRow]: newReports) {
            const auto& source = yuidMessage->GetSource().empty() ? "serp" : yuidMessage->GetSource();
            const auto& subkey = ToString(bsWatchRow->GetTimestamp());
            const auto& value = SerializeFingerprintMatch(*yuidMessage, *bsWatchRow, source);
            writer->AddRow({State.GetFingerprint(), subkey, value}, Output);
        }

    } catch(const yexception& e) {
        writer->AddRow({State.GetFingerprint(), ToString(maxTimestamp), e.what()}, Errors);
    }
}
