#include "new_match_getter.h"

#include <util/generic/hash.h>

using namespace NCrypta::NGraph;

namespace {
    template <typename TIter>
    bool TryNextEvent(TIter& it, time_t minTs, ::google::protobuf::RepeatedPtrField<typename TIter::value_type>& buffer) {
        if (it->GetTimestamp() < minTs) {
            return false;
        }

        buffer.Add()->Swap(&*it);
        ++it;
        return true;
    }

    template <typename TItem>
    void MergeToState(
        ::google::protobuf::RepeatedPtrField<TItem>& state,
        ::google::protobuf::RepeatedPtrField<TItem>& input,
        ::google::protobuf::RepeatedPtrField<TItem>& buffer,
        const ui32 sizeLimit,
        const time_t minTimestamp
    ) {
        buffer.Clear();
        buffer.Reserve(sizeLimit);

        auto stateIt = state.begin();
        auto inputIt = input.rbegin();

        while(static_cast<ui32>(buffer.size()) < sizeLimit && !(stateIt == state.end() && inputIt == input.rend())) {
            bool nextIsState = stateIt != state.end() && (inputIt == input.rend() || inputIt->GetTimestamp() < stateIt->GetTimestamp());
            bool gotEvent = nextIsState ? TryNextEvent(stateIt, minTimestamp, buffer) : TryNextEvent(inputIt, minTimestamp, buffer);

            if (!gotEvent) {
                break;
            }
        }

        state.Swap(&buffer);
        input.Clear();
    }

    time_t GetDelta(const TYuidMessage& yuidMessage, const TParsedBsWatchRow& bsWatchRow) {
        return bsWatchRow.GetTimestamp() - yuidMessage.GetTimestamp();
    }
}

bool TNewMatchGetter::TRememberedMatchEqualTo::operator()(const TRememberedMatch& a, const TRememberedMatch& b) const {
    return a.GetYandexuid() == b.GetYandexuid() && a.GetFpc() == b.GetFpc();
}

size_t TNewMatchGetter::TRememberedMatchHash::operator()(const TRememberedMatch& a) const {
    return ComputeHash(a.GetYandexuid()) + ComputeHash(a.GetFpc());
}

TNewMatchGetter::TNewMatchGetter() {
}

void TNewMatchGetter::AddYuidEvent(TYuidMessage&& row) {
    YuidEvents.Add()->Swap(&row);
}

void TNewMatchGetter::AddBsWatchEvent(TParsedBsWatchRow&& row) {
    BsWatchEvents.Add()->Swap(&row);
}

TNewMatchGetter::TMatches TNewMatchGetter::GetNewReports(TJoinFingerprintsState& state, time_t maxTimestamp, const TConfig& config) {
    const auto& historyMinTs = maxTimestamp - config.GetMaxHistoryAgeSec();
    const auto& matchMinTs = maxTimestamp - config.GetMaxRememberedMatchAgeSec();

    MergeToState(*state.MutableYuidHistory(), YuidEvents, YuidBuffer, config.GetMaxHistorySize(), historyMinTs);
    MergeToState(*state.MutableBsWatchHistory(), BsWatchEvents, BsWatchBuffer, config.GetMaxHistorySize(), historyMinTs);

    auto rememberedMatchSet = GetRememberedMatchSet(*state.MutableRememberedMatches(), matchMinTs);

    auto result = CollectNewMatches(
        state.GetBsWatchHistory(),
        state.GetYuidHistory(),
        config.GetTimeWindowSec(),
        config.GetValidMatchLimit(),
        rememberedMatchSet);

    UpdateRememberedMatches(
        *state.MutableRememberedMatches(),
        rememberedMatchSet,
        config.GetRememberedMatchLimit());

    return result;
}

TNewMatchGetter::TRememberedMatchSet TNewMatchGetter::GetRememberedMatchSet(TRepeatedRememberedMatch& reportedMatches, const time_t minTs) {
    TRememberedMatchSet rememberedMatchSet;
    rememberedMatchSet.reserve(reportedMatches.size());

    for (auto& match: reportedMatches) {
        if (match.GetTimestamp() > minTs) {
            rememberedMatchSet.emplace(std::move(match));
        }
    }

    return rememberedMatchSet;
}

TNewMatchGetter::TMatches TNewMatchGetter::CollectNewMatches(
    const TRepeatedBsWatch& bsWatchHistory,
    const TRepeatedYuids& yuidHistory,
    const time_t timeWindow,
    const size_t validMatchLimit,
    TRememberedMatchSet& rememberedMatchSet
) {
    TMatches matchesToReport;

    auto bsWatchIt = bsWatchHistory.begin();

    for(auto yuidIt = yuidHistory.begin(); yuidIt != yuidHistory.end(); ++yuidIt) {
        for (; bsWatchIt != bsWatchHistory.end() && GetDelta(*yuidIt, *bsWatchIt) >= 0; ++bsWatchIt) {
            if (NeedToReport(*yuidIt, *bsWatchIt, timeWindow, rememberedMatchSet)) {
                matchesToReport.emplace_back(&*yuidIt, &*bsWatchIt);
            }
        }
    }

    if (rememberedMatchSet.size() > validMatchLimit) {
        matchesToReport.clear();
    }

    return matchesToReport;
}

bool TNewMatchGetter::NeedToReport(
    const TYuidMessage& yuidMessage,
    const TParsedBsWatchRow& bsWatchRow,
    const time_t timeWindow,
    TRememberedMatchSet& rememberedMatchSet
) {
    if (GetDelta(yuidMessage, bsWatchRow) > timeWindow) {
        return false;
    }

    RememberedMatchBuffer.Clear();
    RememberedMatchBuffer.SetYandexuid(yuidMessage.GetYandexuid());
    RememberedMatchBuffer.SetFpc(bsWatchRow.GetFpc());
    RememberedMatchBuffer.SetTimestamp(bsWatchRow.GetTimestamp());

    const auto& [it, inserted] = rememberedMatchSet.insert(RememberedMatchBuffer);

    return inserted;
}

void TNewMatchGetter::UpdateRememberedMatches(
    TRepeatedRememberedMatch& output,
    const TRememberedMatchSet& rememberedMatchSet,
    const i64 rememberedMatchLimit
) const {
    output.Clear();
    output.Reserve(rememberedMatchSet.size());

    for (auto& match: rememberedMatchSet) {
        *output.Add() = match;
    }

    if (output.size() > rememberedMatchLimit) {
        std::sort(
            output.pointer_begin(),
            output.pointer_end(),
            [](const auto* a, const auto* b){
                return a->GetTimestamp() > b->GetTimestamp();
            }
        );
        output.erase(output.cbegin() + rememberedMatchLimit, output.cend());
    }
}
