#include "parse_adstat_log.h"
#include "unmask/unmask.h"

#include <crypta/graph/rtmr/lib/common/get_normalized_host.h>
#include <crypta/graph/rtmr/lib/common/make_parsed_bs_watch_row.h>
#include <crypta/graph/rtmr/lib/common/validate.h>
#include <crypta/lib/native/proto_serializer/proto_serializer.h>
#include <crypta/lib/native/tcp_options/tcp_options.h>

#include <contrib/libs/re2/re2/re2.h>
#include <library/cpp/http/cookies/cookies.h>
#include <library/cpp/json/json_reader.h>
#include <util/string/cast.h>
#include <util/string/split.h>

using namespace NCrypta::NGraph;

namespace {
    static const TString KEY_COOKIE{"http_cookie"};
    static const TString KEY_TCP_OPTIONS{"tcp_syn_options"};
    static const TString KEY_UA{"http_user_agent"};
    static const TString KEY_UNIXTIME{"timestamp"};
    static const TString KEY_URL{"request_uri"};

    struct TParsedParams {
        TString Duid;
        TString Domain;
    };

    TMaybe<TString> ExtractDuid(const TMaybe<TString> mask, const TMaybe<TString> maskedValue) {
        if (!mask || !maskedValue) {
            return Nothing();
        }

        return NMask::Unmask(NMask::TMask{*mask}, NMask::TMaskedValue{*maskedValue});
    }

    TMaybe<TString> ExtractDomain(const TMaybe<TString> url) {
        if (!url) {
            return Nothing();
        }

        return GetNormalizedHost(*url);
    }

    TString ExtractYandexuid(TStringBuf queryCookies) {
        THttpCookies cookies(queryCookies);
        static const TString yandexuidKey{"yandexuid"};

        if (cookies.Has(yandexuidKey)) {
            const auto yuid = TString{cookies.Get(yandexuidKey)};
            if (ValidateYandexuid(yuid)) {
                return yuid;
            }
        }

        return {};
    }

    bool IsValidExternalIdentifier(const TString& id, size_t requiredLength) {
        static const THashSet<TString> invalidIds{
            {"ffffffffffffffffffffffffffffffff"},
            {"00000000000000000000000000000000"},
            {"0000000000000000"},
            {"ffffffffffffffff"},
        };
        static const re2::RE2 externalIdentifierMatch(R"(^[0-9a-fA-F]*$)");

        if (id.size() != requiredLength) {
            return false;
        }

        if (invalidIds.count(id) > 0) {
            return false;
        }

        return re2::RE2::FullMatch(id, externalIdentifierMatch);
    }

    TMaybe<TString> ExtractExternalId(TStringBuf tcpOptions) {
        static const ui64 key40 = 40;
        static const size_t key40Length = 32;
        static const ui64 key65 = 65;
        static const size_t key65Length = 16;

        const auto& tryParse = NCrypta::ParseTcpOptions(tcpOptions);
        if (!tryParse) {
            return Nothing();
        }

        const auto& parsed = tryParse->Get();
        if (auto it = parsed.find(key40); it != parsed.end()) {
            if (IsValidExternalIdentifier(it->second, key40Length)) {
                return it->second;
            }
        } else if (auto it = parsed.find(key65); it != parsed.end()) {
            if (IsValidExternalIdentifier(it->second, key65Length)) {
                return it->second;
            }
        }

        return Nothing();
    }

    TMaybe<TParsedParams> ExtractDataFromURL(TStringBuf url) {
        static const TString URL_REF_PARAM{"ref"};
        static const TString URL_MASK_PARAM{"mask"};
        static const TString URL_MASKED_PARAM{"id"};

        TString fakeUrl = "https://host.com/" + TString{url};

        const auto& domain = ExtractDomain(GetParam(fakeUrl, URL_REF_PARAM));
        const auto& duid = ExtractDuid(GetParam(fakeUrl, URL_MASK_PARAM), GetParam(fakeUrl, URL_MASKED_PARAM));
        if (!duid || !domain) {
            return Nothing();
        }

        if (!ValidateDomainUserID(*duid) || !IsCorrectDomain(*domain) || IsYastaticDomain(*domain)) {
            return Nothing();
        }

        return TParsedParams{.Duid = *duid, .Domain = *domain};
    }
}

namespace NCrypta::NGraph {
    TParseAdstatLog::TParseAdstatLog(const TResources& resources)
        : Resources(resources)
    {
    }

    TParseAdstatLog::TParseResult TParseAdstatLog::ParseMessage(const TStringBuf record,
        const uatraits::detector& detector) const {
        try {
            NJson::TJsonValue parsed;
            NJson::ReadJsonTree(record, &parsed, true);

            const auto& cookies = parsed[KEY_COOKIE].GetString();
            const auto& url = parsed[KEY_URL].GetString();
            const auto& ua = parsed[KEY_UA].GetString();
            const auto& tcp_options = parsed[KEY_TCP_OPTIONS].GetString();
            const auto& timestamp = static_cast<ui32>(FromString<double>(parsed[KEY_UNIXTIME].GetString()));
            const auto& traits = detector.detect(ua);

            if (!ValidateUATraits(traits)) {
                return {.ErrorCode = EParseResult::NotItp};
            }

            const auto& parsedParams = ExtractDataFromURL(url);
            const auto& extId = ExtractExternalId(tcp_options);

            if (!parsedParams || !extId) {
                return {.ErrorCode = EParseResult::Invalid};
            }

            auto message = MakeBsWatchRow(parsedParams->Duid, parsedParams->Domain, traits, timestamp, ExtractYandexuid(cookies), "adstat");
            return {.ErrorCode = EParseResult::Ok, .ExtId = *extId, .Message = NProtoSerializer::ToString(message)};

        } catch (const yexception& e) {
            return {.ErrorCode = EParseResult::Error, .Message = e.what()};
        }
    }

    void TParseAdstatLog::ParseRow(const TReader::TRowType& row,
        const uatraits::detector& detector,
        TWriter* writer) const {
        for (const auto& line : StringSplitter(row.Value).Split('\n').SkipEmpty()) {
            const auto& parseResult = ParseMessage(line.Token(), detector);
            switch (parseResult.ErrorCode) {
                case EParseResult::Ok: {
                    writer->AddRow({parseResult.ExtId, row.SubKey, parseResult.Message}, EOutputTable::Join);
                    break;
                }
                case EParseResult::Error: {
                    writer->AddRow({row.Key, row.SubKey, parseResult.Message}, EOutputTable::Errors);
                    break;
                }
                case EParseResult::NotInitialized: {
                    writer->AddRow({row.Key, row.SubKey, "Not initialized reply from ParseMessage"}, EOutputTable::Errors);
                    break;
                }
                default:
                    break;
            }
        }
    }

    void TParseAdstatLog::Do(TReader* reader, TWriter* writer) {
        auto resourceHolder = Resources.GetHolder();
        const auto& detector = resourceHolder.GetUatraitsDetector();

        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();

            ParseRow(row, detector, writer);
        }
    }
}
